#!/usr/bin/python

## @package ardi.consolidator.factory
#
# Factory methods to launch the consolidator daemon from Twistd.

from twisted.application import internet,service
from twisted.internet.protocol import ServerFactory
from twisted.python import log
from twisted.application.service import IProcess

#import logging

import sys

sys.path.append("/opt/ardi/srv/consolidator")

import time
import ardi.consolidator.incoming as incoming
import ardi.consolidator.subscription as subscription
import threading
import argparse
import logging
import requests
import xmltodict
import traceback

## Message Class
#
#  This represents a single OOB message
class Message:
    def __init__(self):
        self.time = 0
        self.name = ""
        self.content = ""

## Consolidator Core Class
#
#  This core object holds all of the common information the server requires to function.
class Core:
    def __init__(self):
        ## A list of ARDI points
        self.points = {}
        ## The last update times for these points
        self.pointimes = {}
        ## A mutual-exclusion lock to protect the lists in a multi-threaded environment
        self.lock = threading.Lock()
        ## The time of the last data update
        self.lastupdate = 0
        ## A list of active subscriptions to the data
        self.subscriptions = []
        ## The number of users allowed to access live data simultaniously.
        self.maxusers = 0
        ## The message queue
        self.messagequeue = []
        self.debug=0

    ## Requests server information
    #
    #  This gets license details from the ARDI server to determine the number of users
    #  who are permittied to access live data simultaniously.
    #
    #  @param srv The URL of the server
    def FetchServerInfo(self,srv):
        success = False
        #while success == False:
        try:
            print("Checking http://" + srv + "/api/connect for server details...")
            r = requests.get('http://' + srv + "/api/connect")
            parts = xmltodict.parse(r.text)
            try:
                for bt in parts['ardi']['setting']:
                    if bt['@name'] == 'users':
                        if bt['#text'] != 'Unlimited':
                            try:
                                self.maxusers = int(bt['#text'])
                            except:
                                pass
            except:
                pass
            success = True
        except:
            traceback.print_exc()
            #time.sleep(10)
            pass

        logging.info('Connected to Server ' + str(srv) + " - " +str(self.maxusers) + " Concurrent User Max")
        print('Connected to Server ' + str(srv) + " - " +str(self.maxusers) + " Concurrent User Max")
        #print "Server: " + str(srv)

    ## Handle New Data
    #
    #  This function is called every time new data arrives.
    #
    #  @param name The name of the data point.
    #  @param value The value of the data point
    def NewDataArrived(self, name, value):
        
        self.lock.acquire()
        
        value = value.strip()

        if name == "":
            self.lock.release()
            return
        
        #Handle an OOB notification message
        if name[0] == "!":

            ctime = time.time()
            msg = Message()
            msg.time = ctime
            msg.name = name
            msg.value = value
            self.messagequeue.append(msg)

            #Remove any messages 10 seconds old or older.
            try:
                for m in self.messagequeue[:]:
                    if ctime - m.time > 10:
                        print("Purging Old Records")
                        self.messagequeue.remove(m)
            except:
                print("Failed Purging Records")
            self.lock.release()
            return
        
        try:
            if self.points[name] != value:
                self.points[name] = value
                self.pointimes[name] = time.time()
        except:
            self.points[name] = value
            self.pointimes[name] = time.time()
            
        self.lock.release()
        self.lastupdate = time.time()

class IncomingService(service.Service):
    def __init__(self, core):
        self.core = core
        pass
    
    def startService(self):
        service.Service.startService(self)
        pass

class OutgoingService(service.Service):
    def __init__(self,core):
        self.core = core
        pass
    
    def startService(self):
        service.Service.startService(self)
        
class IncomingFactory(ServerFactory):
    protocol = incoming.Submission
    def __init__(self, service):
        self.service = service
        
    def buildProtocol(self, addr):
        return incoming.Submission(self)

## Called by Twistd To Setup Consolidator
#
#  This function is called by the Twistd daemon to prepare the consolidator.
#
#  It launches both the incoming and outgoing TCP services and places them
#  in a Twistd service hierarchy.
def ConsolidatorFactory(inport,outport,srv):
    
    core = Core()
    
    core.FetchServerInfo(srv)
    
    top_service = service.MultiService()

    inserv = IncomingService(core)
    inserv.setServiceParent(top_service)

    infact = IncomingFactory(inserv)
    tcp_service = internet.TCPServer(int(inport),infact)
    tcp_service.setServiceParent(top_service)

    site = subscription.GetService(core)
    tcp_service = internet.TCPServer(int(outport),site)
    tcp_service.setServiceParent(top_service)

    return top_service
        
