import sys
import pycurl
import xmltodict
import time
import socket
import ardi.driver.transform as transform
import logging
import logging.handlers
from subprocess import Popen
import subprocess
import platform
import traceback
try:
	import thread
except:
	import threading as thread
import sys
import requests

## @package ardi.driver.drivercore
#  ARDI Live Driver Connection
#
#  The DriverCore class acts as a connection to a single data source within a DriverBase.
#
#  Each core is allocated its own thread to avoid issues with one device causing problems with others.


try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode

#from StringIO import StringIO

## Data Point Information
#
#  An empty class used for storing information about data points
class pointinfo:
        pass

## A connection to a single instance or device
#
#    Each core in the ARDI driver architecture is used to connect to a single instance or device
#
#    This class contains the common behaviours between different drivers. The job of connecting, querying and
#    otherwise interacting with the actual data source is up to the driver.
class ardicore:

    ## Initialises the Core
    def __init__(self,driver):

        ##The URL to send all information to (drivers may be remote)        
        self.url = "localhost"

        ##The source asset ID
        self.asset = 1

        ##A copy of the actual driver class (the code interacting with the data storage)
        self.driver = driver

        ##True of the driver is paused, False otherwise
        self.paused = False

        ##True if the driver is handling "Interest Based Data Loading"
        self.interest = False

        ##An internal flag used to shutdown the driver if requested
        self.shutdown = False

        ##The current amount of time to wait before attempting to reconnect
        self.reconnecttime = 10

        ##A reference to the driver core
        driver.core = self

        ##This flag is 'True' when optimisation is required
        self.optimiserequired = True

        ##A fast lookup dictionary to find the internal ARDI code for an address
        self.codelookup = {}

        ##The number of seconds between samples
        self.sampletime = 10

        ##Used to flag a recovery attempt after the driver closed a connection due to error.
        self._recoverystep = 0

    ## Set Connection Details
    #
    #    This function sets some of the basic identity information about the core.
    #
    #    \param url: The URL to query to get core configuration information from ARDI
    #    \param assetid: The source ID that this core represents
    def SetARDIDetails(self, url, assetid):        
        self.url = url
        self.asset = assetid
        self.InitLogger()

    ## Set Driver
    #
    #    This function associates the given driver with this core
    #
    #    \param driver: Associate this core with a driver          
    def SetARDIDriver(self, driver):  
        self.driver = driver
        self.driver.asset = self.asset
        driver.core = self

    ## Initialises the logging for this indiviual source
    def InitLogger(self):

        if hasattr(self,'logger'):
            return
        
        #print 'Initialising Logger for ' + self.asset
        logfile = '/var/log/ardi/source-' + self.asset + '-' + self.info.profile + '-live.log'
        if (platform.system() == 'Windows'):
                    logfile = "c:\\windows\\temp\\ardi-source-" + self.asset + '-' + self.info.profile + "-live.log"

        ##The logger for a specific data source, rather than the driver process as a whole
        self.logger = logging.getLogger('Source '+self.asset)  
        if (len(self.logger.handlers) == 0):
            handler = logging.handlers.WatchedFileHandler(logfile)
            handler.setLevel(logging.INFO)
            formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            handler.setFormatter(formatter)
            self.logger.addHandler(handler)
            self.logger.info('----DRIVER STARTED----')
            self.logger.info('Starting Data Source ' + self.asset + " on thread " + str(thread.get_ident()))

    ## Pause the core    
    def Pause(self):
        self.logger.warning("Service Paused")
        self.paused = True

    ## Resume a paused core        
    def Resume(self):
        self.logger.warning("Service Resumed")
        self.paused = False

    ## Force the core to reload
    #
    #    This function is usually called after a change to the required data
    def Reload(self):
        self.LoadConfig()

    ## Stops the source
    #
    #    This function is called as part of a driver restart or shutdown
    def Stop(self):
        self.shutdown = True
        self.logger.info('Shutdown Requested')

    ## Change the driver update rate
    #
    #    This function varies the delay between data refreshes.
    #
    #    \param rate: The new delay (in seconds) between samples"""
    def SetUpdateRate(self,rate):
        self.sampletime = float(rate)
        self.logger.info("Changing Sample Rate: " + str(self.sampletime) + " seconds")

    ## Reload driver configuration
    #
    #    This function triggers the driver to reload it's details from the ARDI server. Note that this
    #    is very different from the same function located in the DriverBase class. The data returned by
    #    this query includes all of the individual data points to be sampled from the data source.
    def LoadConfig(self):
        if (self.source != ''):
            self.logger.info("Loading Cached Settings @ " + self.source)
            fl = open(self.source,'r')
            xml = fl.read()
            fl.close()
            self.parseconfig(xml)
            return

        #response = StringIO()
        
        fullurl = "http://" + self.url + "/api/sourcedata?asset=" + str(self.asset) + "&mode=" + str(self.info.mode) + "&profile=" + str(self.info.profile)

        self.logger.info("Requesting Settings From " + fullurl)
        web = requests.get(fullurl)		
        #try:
        #    c = pycurl.Curl()
        #    c.setopt(c.URL,fullurl)
        #    c.setopt(c.WRITEFUNCTION,response.write)
        #    c.setopt(c.TIMEOUT,5)
        #    c.setopt(c.CONNECTTIMEOUT,5)
        #    c.perform()
        #finally:
        #    c.close()

        self.parseconfig(web.text)

    ## Parse the configuration details
    #
    #    This function processes the XML from 'LoadConfig' to setup the driver and the many data points.
    #
    #    \param xml: The XML configuration data to parse"""
    def parseconfig(self,xml):
        #print "XML: " + xml
        result = xmltodict.parse(xml)
        
        #print result
        
        #Core Parameters
        ##The original address of this particular data source
        self.driveraddr = result['sourcedata']['connection']['address']

        ##The IPC address of the data source, to confirm details
        self.hostport = int(result['sourcedata']['connection']['port'])
        self.driver.SetAddress(self.driveraddr)

        if self.driveraddr is None:
            self.driveraddr = ""

        #Data Points
        try:
            ##The list of all data points to load from this source
            self.pointlist = []
            pointset = result['sourcedata']['points']['point']
            if '@address' in pointset:
                pointset = [pointset]
            
            for point in pointset:
                #Add Data Points to List
                newpoint = pointinfo()
                newpoint.address = point['@address']
                newpoint.code = point['@code']
                newpoint.transform = point['@transform']
                newpoint.changed = False
                newpoint.value = ""
                newpoint.intvalue = ""
                newpoint.lookupid = 0
                try:
                    newpoint.lookupid = int(point['@lookup'])
                except:
                    pass
                self.pointlist.append(newpoint)
        except:
            pass
        
        if self.interest == True:
            #Interest-based Data Loading
            ##A backup of the list of ALL data points, when interest-based loading is enabled
            self.allpoints = self.pointlist
            self.pointlist = []
            
        self.optimiserequired = True

        sleeptimeset = False
        sleeptime = 5
        if hasattr(self.driver,'pollinterval'):
            sleeptime = self.driver.pollinterval

        try:
            sleeptime = result['sourcedata']['connection']['samplerate']
        except:
            pass

        self.sampletime = float(sleeptime)

        self.logger.info("Total Inputs: " + str(len(self.pointlist)))
        self.logger.info("Using Sample Delay: " + str(self.sampletime) + " seconds")

        try:
            self.driver.Remap()
        except:
            pass

        #Rebuild code lookup table...
        self.codelookup = {}
        x = -1
        for itm in self.pointlist:
            x = x + 1
            self.codelookup[itm.code] = x

        self.addresslookup = {}
        for itm in self.pointlist:
            if itm.address not in self.addresslookup:
                self.addresslookup[itm.address] = []
            self.addresslookup[itm.address].append(itm)

        #print self.codelookup
        #self.InitLogger()
        #print "---------------"
        #print str(self.addresslookup)
    
    def RegisterInterest(self,code,subscribe):
        """Register a new set of interest points

        This function is used for interest-based data loading, which optimises the scanning of data so
        that it only occurs when someone is actually interested (ie. looking) for it.

        This function triggers a new call to the drivers 'Optimise' function.

        \param code: A new set of points to use during data updates

        """
        for point in self.pointlist:
            if (point.code == code):
                if (subscribe == True): 
                    point.timeout = time.time() + 30
                else:
                    point.timeout = time.time() - 1
                return
            
        for point in self.allpoints:
            if (point.code == code):
                if (subscribe == True): 
                    point.timeout = time.time() + 30
                else:
                    point.timeout = time.time() - 1
                self.pointlist.append(point)
                return

    ## Parse the configuration details
    #
    #    This function processes the XML from 'LoadConfig' to setup the driver and the many data points.
    #
    #    \param xml: The XML configuration data to parse 
    def RunLoop(self):
        """The main driver loop

        This function is called repeatedly and makes up the main 'loop' of the driver. It is responsible for keeping
        the driver connected, managing reconnection attempts, re-optimising the driver where required and polling
        for new information."""
        if self.driver.connected == False:
            self.logger.info("Connecting To: " + self.driveraddr)

            if (self.logger.getEffectiveLevel() == logging.DEBUG):
                st = ""
                for itm in self.pointlist:
                    if st != "":
                        st = st + ", "
                    st = st + itm.address
                    
                self.logger.debug("Points: " + st)
        
            res = self.driver.Connect()
            if (res == True):
                self.logger.info("Connected To Target Device")
                self.reconnecttime = 10
            else:
                self.logger.error("Failed To Connect - " + str(self.reconnecttime) + " seconds to next attempt")
                time.sleep(self.reconnecttime)
                self.reconnecttime *= 2
                if self.reconnecttime > 6000:
                    self.reconnecttime = 6000
                return
        
        if self.driver.polling == True:
            #This is a polling driver - it will need constant re-scanning            
            self.scanpollingdriver()            
        else:
            #This is a non-polling driver - it will let us know when new data arrives
            self.scansubscriptiondriver()

        if self._recoverystep > 1:
            time.sleep(self.reconnecttime)

    ## Scan a 'Polling' Driver
    #
    #    This function is used to scan a driver that expects regular polling - a 'pull' driver"""
    def scanpollingdriver(self):
        if self.optimiserequired == True:
            try:
                self.driver.Optimise()
                self.optimiserequired = False
            except Exception as e:
                self.exception('Failed To Optimise: ' + str(e))
                return

        pollstart = time.time()
        if self.driver.connected == True and self.paused == False:
            try:                
                self.driver.Poll()
                self.pushdataupdates()
            except Exception as e:
                self.exception('Failed To Poll: ' + str(e))
                self.optimiserequired = True
                try:
                    self.driver.Disconnect()
                except:
                    pass
                self.driver.connected = False

            self._recoverystep = 0
            timetaken = time.time() - pollstart
            #print "Sampling Took " + str(timetaken) + " seconds"
            if (timetaken < self.sampletime):
                time.sleep(self.sampletime - timetaken)            
        return

    ## Scan a 'Subscription' Driver
    #
    #    This function is used to scan a driver that pushes data to the client rather than requiring regular polling"""
    def scansubscriptiondriver(self):
        if self.optimiserequired == True:
            self.driver.Resubscribe()
            self.optimiserequired = False
            
        if self.driver.iswaiting == False:
            self.driver.Subscribe()
            return

    def exception(self, msg):
        if (self.logger.getEffectiveLevel() == logging.DEBUG):
            self.logger.exception(msg)
        else:
            self.logger.error(msg)

    ## Updates a data point given the source address rather than the ARDI address
    #
    #    This allows drivers to optimise by querying for a field ONCE rather than multiple times, and have
    #    one source data point map to multiple ARDI points with different transforms."""
    def NewData(self, address, value):

        #print(str(self.addresslookup))

        if address in self.addresslookup:
            set = self.addresslookup[address]
            for itm in set:
                if (itm.intvalue != value):
                    vx = value
                    try:
                        if value != "^":                            
                            if itm.lookupid != 0:
                                try:                                
                                    lid = itm.lookupid
                                    vx = self.base.lookups[lid][str(vx)]
                                except:
                                    #traceback.print_exc()
                                    vx = "None / Unknown"                        
                    except:
                        pass

                    if itm.intvalue == vx:
                        continue

                    itm.intvalue = vx

                    try:
                        tx = transform.transform(itm.transform)
                        itm.value = tx.run(vx)
                        #print str(itm.intvalue) + " = " + str(itm.value)
                    except:                        
                        itm.value = itm.intvalue
                    
                    itm.changed = True
                    if (self.interest == True):
                        if time.time() > itm.time:
                            self.pointlist.remove(itm)     

    ## Updates a data point
    #
    #    This function stores the new value of a data point and prepares it for transmission to ARDI.
    #
    #    \param code: The raware-style code for the data point (ie <assetid>:<propertyid>:<inputname>)
    #    \param value: The untransformed value for of the data point"""
    #
    #   @deprecated This function has been deprecated in favour of NewData - it still remains for compatibility reasons.
    def UpdateData(self, code, value):
        if code in self.codelookup:
            indx = self.codelookup[code]
            itm = self.pointlist[self.codelookup[code]]            
            if (itm.intvalue != value):
                self.pointlist[indx].intvalue = value
                #print "Updated " + self.pointlist[indx].intvalue + " to " + value
                try:
                    tx = transform.transform(itm.transform)
                    itm.value = tx.run(value)
                    #print str(itm.intvalue) + " = " + str(itm.value)
                except:
                    #print "Exception Transforming"
                    itm.value = itm.intvalue
                
                #print "Tx: " + str(itm.intvalue) + " to " + str(itm.value)
                itm.changed = True
                if (self.interest == True):
                    if time.time() > itm.time:
                        self.pointlist.remove(itm)                

    ## Updates a data point
    #
    #    This function pushes all of the data changes from the last loop to ARDI.
    #
    #    Note that if the base and shell properties are set, this function will also call a
    #    script as part of it's execution, sending all of the new values to it via stdin
    #    as '<name>=<value>[CR][LF]' pairs. 
    #
    #    \param code: The ARDI code for the data
    #    \param value: The value of the data"""
    def pushdataupdate(self,code,value):

        #Before services were available, this allows you to install a script to be called on data changes.
        #if self.basescript != '':
        #    if self.baseshell != '':
        #        process = Popen([self.baseshell, self.basescript], shell=False, stdin=subprocess.PIPE, stdout=None,stderr=None)
        #    else:
        #        process = Popen([self.basescript], shell=False, stdin=subprocess.PIPE, stdout=None,stderr=None)			
        #    for itm in self.pointlist:
        #        if (itm.changed == True):
        #             process.stdin.write(itm.code + "=" + itm.value + "\r\n")										
        #process.stdin.write(".\r\n")
        #process.stdin.close()

        self.logger.debug("Pushing Updated Data")
         
        #Send Web Request for ARDI Server
        try:
            #response = StringIO()

            fullurl = "http://" + self.url + "/api/sourcepush"

            anydata = False
            post_data = {'asset': self.asset, 'port': self.hostport}
            for itm in self.pointlist:
            	if (itm.changed == True):
            		post_data[itm.code] = itm.value
            		anydata = True
            if anydata == True:
                postfields = urlencode(post_data)
                #print postfields

                #try:
                #    c = pycurl.Curl()
                #    c.setopt(c.URL,fullurl)
                #    c.setopt(c.WRITEFUNCTION,response.write)
                #    c.setopt(c.TIMEOUT,2)
                #    c.setopt(c.CONNECTTIMEOUT,1)
                #    c.setopt(c.POSTFIELDS,postfields)
                #    c.perform()
                #    if response.getvalue() != "OK":
                #        self.logger.info("DAQ Problem: " + response.getvalue())
                #finally:
                #    c.close()
                web = requests.post(fullurl,data=post_data,timeout=1000)
            #self.logger.info("Web Response: " + response.getvalue())
        except Exception as e:
            if e[0] != 28:
                self.logger.error('Error Sending to Website ' + fullurl + ': ' + str(e))
            pass
			
        try:
            #Send Socket Request for Consolidation Server
            self.logger.debug("Updating Consolidation Server")

            ##The socket connection to the consolidator
            self.client = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
            self.client.connect((self.info.subhost,int(self.info.subport)))
            for itm in self.pointlist:
            	if (itm.changed == True):
                	#print itm.code + "=" + str(itm.value)
                        if (sys.version_info > (3, 0)):
                                self.client.send(bytes(itm.code + "=" + str(itm.value) + "\r\n",'utf-8'))
                        else:
                                self.client.send(itm.code + "=" + str(itm.value) + "\r\n")
                	#self.logger.warning(itm.code + "=" + str(itm.value))
                	#self.client.flush()					
            self.client.close()
            self.logger.debug("Update Complete!")
        except:
            self.logger.error('Error Sending to Consolidation Server @ ' + self.info.subhost + ":" + str(self.info.subport))
            traceback.print_exc()
            pass

        for itm in self.pointlist:
            itm.changed = False

    ## Sends data updates if required
    #
    #    This function checks through all of the current data to see if a data update is required, and
    #    starts the update process if it is."""
    def pushdataupdates(self):
        doit = False
        for itm in self.pointlist:
            if (itm.changed == True):
                doit = True
                break
        if doit == True:
            #self.logger.info('Sending Data Updates')
            self.pushdataupdate("","")                

    ## Starts the driver
    #
    #    This function initiates the driver.
    #
    #    \param address: The encoded destination address (ie. IP address) of the device
    #    \param asset: The asset id that this source represents
    def Launch(self,address,asset):
        self.SetARDIDetails(address,asset)
        self.LoadConfig()
        self.Start()
        #self.logger('Thread ' + str(thread.get_ident()) + " Started For Asset " + asset)

    ## Launch individual core web service
    #
    # @deprecated Removed due to poor IPC performance when running as a Linux service
    def launchwebservice(self):
        t = threading.Thread(target=webservice, args = (self,))
        t.daemon = True
        t.start()

    ## Main Thread
    #
    #   This is the main thread for the source. It continues until shutdown is requested.
    def Start(self):
        while True:       
            #self.logger.info('Refreshing Thread ' + str(thread.get_ident()))
            if self.shutdown == True:
                self.logger.info('Shutting Down Source Thread ' + str(thread.get_ident()))
                break
            self.RunLoop()
