## @package ardi.driver.histdriverbase
#  ARDI Historian Driver Service
#
#  The 'Base' class encapsulates all of the essential startup and control behaviours of an ARDI historical driver.
#
#  Each 'Base' contains multiple 'Core' objects, which in turn contain a single source instance.
#
import os
import platform
import time
import sys
import logging
import logging.handlers
import threading
import ardi.driver.historianipc as historianipc
import ardi.driver.histdrivercore as histdrivercore
import argparse
#import pycurl
import xmltodict
import traceback
import requests
#from StringIO import StringIO
from twisted.web import server,http,resource

class iologger:
    def __init__(self,logger,level):
        self.logger = logger
        self.linebuf = ''
        self.log_level = level
        
    def write(self,buf):
        for line in buf.rstrip().splitlines():
            self.logger.log(self.log_level,line.rstrip())

## Structure for source details
#
#  This class is used as a structure to hold details about a single ARDI data source.
class ardisourcedetail:
    def __init__(self):
        self.id = 0
        self.subport = 5336
        self.subhost = "localhost"
        self.mode = 0
        self.profile = 1

def corethread(core, servername, assetid):
    core.Launch(servername, assetid)

## Starts the webservice used to communicate with and manage this driver
#
# \param core: The base driver that is controlled by this web service
def webservice(core):
    
    core.logger.info('Starting IPC Service')    
    
    ipc = historianipc.DriverIPC()
    ipc.core = core
    ipc.port = core.hostport
    if ipc.port == 0:
        ipc.port = 8081
        
    try:
        ipc.start()
    except KeyboardInterrupt:
        pass

## The base class for a historian driver.
#
#    This class is responsible for all stages of driver life-cycle.
#
#    It also communicates directly with ARDI to get details on what data sources it should serve via HTTP web requests
class historian:

    ## Initialises the historian
    def __init__(self):
        ##The IPC port for this driver
        self.hostport = 10000
        self.assets = []
        pass
    
    ## Prints usage information if startup fails
    def usage(self):
        print("Historian Startup: <drivername> <asset1>,<asset2>...<assetn>")
        return

    ## Initialises the historian logger
    def initlogger(self):

        logging.basicConfig(level=logging.INFO)
        ##The logger used to record information about the driver process as a whole
        self.logger = logging.getLogger(__name__)
        
        fname = "/var/log/ardi/driver-" + str(self.hostport) + ".log"
        if (platform.system() == 'Windows'):
            fname = "c:\\windows\\temp\\ardi-driver-" + str(self.hostport) + ".log"

        loghandler = logging.handlers.WatchedFileHandler(fname)
        loghandler.setLevel(logging.DEBUG)

        # create a logging format
        
        formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        loghandler.setFormatter(formatter)
        
        # add the handlers to the logger
        
        self.logger.addHandler(loghandler)

        self.logger.info('----HISTORIAN RESTART----')
        self.logger.info('Starting Historian Process')
        
        redir = iologger(self.logger,logging.ERROR)
        sys.stderr = redir

    ## Start the Driver
    #
    #    This function is called once all initialisation is complete. It launches sources and the IPC service.
    #
    #   \param driverfactory: The factory class that produces driver instances
    def start(self,driverfactory):
        self.initlogger()

        ##A class that generates historian drivers on request
        self.factory = driverfactory

        arglist = sys.argv
        if len(arglist) >= 2:
            if arglist[1] == 'start':                
                arglist = arglist[2:]
            else:
                arglist = arglist[1:]
  
        parser = argparse.ArgumentParser(description='ARDI Driver')
        parser.add_argument('driverid',metavar='PortNo',type=int, help='Unique ID/port for the driver', nargs='?')
        parser.add_argument('server',metavar='Server', help='Server Name', nargs='?', default='localhost')
        parser.add_argument('assetid',metavar='AssetNo',type=int, help='Asset IDs to act as a driver for', nargs='?')
        parser.add_argument('--multiple',dest='multi',action='store_const',const=True,default=False,help='Allows this driver to connect to multiple devices')        
        parser.add_argument('--script','-r',dest='script',default='')
        parser.add_argument('--shell','-l',dest='shell',default='')
        parser.add_argument('--config','-c',dest='configfile',default='')        
        parser.add_argument('--debuglevel',dest='debuglevel',default=2)
        args = parser.parse_args(arglist)
        
        self.logger.debug(args)        
        
        servername = args.server
        self.hostport = int(args.driverid)

        self.servername = args.server
        
        self.logger.info('Driver ' + str(self.hostport) + " feeds ARDI server " + servername)

        #self.script = args.script
        #self.shell = args.shell        

        ##A list of the currently active ARDI sources
        self.drivers = {}

        ##A quick lookup dictionary to map addresses to ARDI names
        self.assetids = []
        if args.assetid != None:
            for assetid in args.assetid:
                self.assetids.append(assetid)
        else:
            while True:
                try:
                    #response = StringIO()
                
                    fullurl = "http://" + servername + "/api/driverdata?port=" + str(self.hostport) + "&mode=1"
                
                    #c = pycurl.Curl()
                    #c.setopt(c.URL,fullurl)
                    #c.setopt(c.WRITEFUNCTION,response.write)
                    #c.setopt(c.TIMEOUT,5)
                    #c.setopt(c.CONNECTTIMEOUT,5)
                    #c.perform()
                    #c.close()
                    web = requests.get(fullurl,timeout=10)
                    break
                except:
                    time.sleep(10)
                    pass
            #print response.getvalue()

            self.parseconfig(web.text)
            
            self.logger.info( "Load Default Configuration From Server")

        #for assetid in self.assetids:
        for asst in self.assets:
            assetid = asst.id
            self.logger.info("Creating historian driver instance for " + str(assetid))
            instance = driverfactory.createinstance()
            core = histdrivercore.historiancore(instance)
            core.base = self
            core.source = args.configfile
            core.profile = asst.profile
            self.drivers[str(assetid) + "_" + str(core.profile)] = core
            core.Launch(servername,assetid)

        #print self.drivers
        webservice(self)

    ## Imports the configuration XML file
    #
    #    This function loads the list of data points within the XML text provided. The function is usually called from LoadConfig
    #
    #    \param xml: The configuration data in XML format"""
    def parseconfig(self,xml):
        result = xmltodict.parse(xml)
                        
        #Core Parameters
        try:            
            if '@id' in result['driverdata']['source']:
                ast = ardisourcedetail()
                ast.id = result['driverdata']['source']['@id']
                ast.profile = result['driverdata']['source']['@profile']
                ast.mode = result['driverdata']['source']['@mode']
                self.assets.append(ast)
                
                self.assetids.append(result['driverdata']['source']['@id'])
            else:
                for itm in result['driverdata']['source']:
                    ast = ardisourcedetail()
                    ast.id = itm['@id']
                    ast.profile = itm['@profile']
                    ast.mode = itm['@mode']
                    self.assets.append(ast)
                
                    self.assetids.append(itm['@id'])
        except:
            pass
            
        self.lookups = {}
        #print(str(result))
        try:
            if result['driverdata']['lookups'] == 'present':
                self.loadLookups()
        except:
            pass

    ## Load lookup tables
    #
    #  If the ARDI server uses lookup tables, this function downloads copies of all tables.
    #
    #  This allows the driver to perform lookup translation, sparing the web server excessive usage.
    def loadLookups(self):
        self.logger.info("Loading Lookup Table")
        try:
            response = StringIO()
        
            fullurl = "http://" + self.servername + "/api/getlookuptables"
        
            c = pycurl.Curl()
            c.setopt(c.URL,fullurl)
            c.setopt(c.WRITEFUNCTION,response.write)
            c.setopt(c.TIMEOUT,5)
            c.setopt(c.CONNECTTIMEOUT,5)
            c.perform()
            c.close()
        except:
            self.logger.exception("Failed to load lookup values from server")
            return

        #print response.getvalue()

        currentid = 0
        lines = response.getvalue().splitlines()
        for l in lines:
            bits = l.split("\t")
            if len(bits) == 1:
                bits = l.split(",")
            if len(bits) == 1:
                if l[0] == '!':
                    currentid = int(l[3:])
                    continue
            else:
                if bits[1] == "":
                    continue
                try:
                    self.lookups[currentid][bits[0]] = bits[1]
                except:
                    self.lookups[currentid] = {}
                    self.lookups[currentid][bits[0]] = bits[1]
                    #self.logger.info("  Adding Lookups for Property")
        #print(str(self.lookups))

    ## Performs a query on a single source
    #
    #    \param assetid: The SOURCE asset id
    #    \param points: A list of points to query
    #    \param start: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param end: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param function: The function to perform. Functions include 'raw', 'avg', 'sum', 'min' and 'max'
    #    \param grain: The number of seconds between samples, or '0' for natural sampling rate
    #    \param first: True if this the data is to include the samples immediately preceeding the start time.
    def Query(self,assetid, points,start,end,function,grain,first,context=1):
        print("Querying Context: " + str(assetid) + '_' + str(context))
        return self.drivers[str(assetid) + '_' + str(context)].Query(points,start,end,function,grain,first)

    ## Forces a historian source to reload
    #
    #    This function is usually called internally by a change to the ARDI data.
    #
    #    \param assetid: The SOURCE asset id            
    def Reload(self, assetid,context=1):
        self.drivers[str(assetid) + "_" + str(context)].Reload()

    ## Setup a new historian.
    #
    #    This function is usually called internally by a change to the ARDI data.
    #
    #    \param assetid: The SOURCE asset id    
    def Spinup(self,assetid,context=1):
        try:
            self.drivers[str(assetid) + "_" + str(context)].Reload()
            self.logger.info("Reloaded historian source " + str(assetid) + " due to configuration change")
        except:
            self.logger.info("Spinning up historian driver instance for " + str(assetid))
            instance = driverfactory.createinstance()
            core = histdrivercore.historiancore(instance)
            core.base = self
            core.source = args.configfile
            core.profile = asst.profile
            self.drivers[str(assetid) + "_" + str(asst.profile)] = core
            core.Launch(servername,assetid)

    ## Forces the entire driver process to reload
    #
    #   @todo Note that this is currently unimplemented for historian drivers!
    #
    def Restart(self):
        
        self.logger.info('Driver Restart Requested via IPC')
        #self.StopDrivers()
        #self.LoadConfig()
