import pycurl
import xmltodict
import time
import socket
import transform
import platform
from subprocess import Popen
import subprocess
import logging
import logging.handlers
import thread
import os

try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode

from StringIO import StringIO

class pointinfo:
    """Data Point Information

    An empty class used for storing information about data points"""
    pass

class historianquery:
	def __init__(self,cr,fun,endtime):
		self.core = cr
		self.function = fun
		self.lastvalues = {}
		self.lasttimestamps = {}
		self.grain = 2
		self.ends = endtime

	def Cleanup(self,stamp):
	    vv = ""
	    if (self.function == "changes"):
	        #In 'Changes' mode, we need to output any resudual values...
	        for keys in self.lasttimestamps:
	            if self.lasttimestamps[keys] != False:
	                vv += str(keys) + "," + str(stamp) + "," + str(self.lastvalues[keys]) + "\r\n"

	    return vv

	def AddRecord(self, id, stamp, value, addr):
							
		if self.core.transforms[addr] != "":
		    tx = transform.transform(self.core.transforms[addr])
		    #print "Transforming: " + self.core.transforms[addr]
		    value = tx.run(value)

		if (self.function == "changes"):
			if id not in self.lastvalues:
				self.lastvalues[id] = value
				self.lasttimestamps[id] = False
			else:
				#print "Comparing Against " + str(value)
				if (self.lastvalues[id] == value):
					self.lasttimestamps[id] = stamp
					#print "Skipping Line!"
					return ""
				else:	
					oldstamp = self.lasttimestamps[id]
					self.lasttimestamps[id] = False
					oldvalue = self.lastvalues[id]
					self.lastvalues[id] = value
						
					if oldstamp != False:
						return str(id) + "," + str(oldstamp) + "," + str(oldvalue) + "\r\n" + str(id) + "," + str(stamp) + "," + str(value) + "\r\n"

		return str(id) + "," + str(stamp) + "," + str(value) + "\r\n"

class historiancore:
    """A connection to a single instance or device

    Unlike live drivers, historian drivers are mostly passive and don't run their own individual threads.

    They wait until they are queried by the system to return data.
    """
    
    def __init__(self,driver):
        """ Initialises the Core"""
        ##The url to return data to, used for remote drivers
        self.url = "localhost"

        ##The source ID
        self.asset = 1

        ##The actual driver instance - the class that performs the query on the data store
        self.driver = driver

        ##A self-reference        
        driver.core = self

        ##A number to track any dynamic addresses (usually associated with random data)
        driver.dynaddress = 0

        self.qry = historianquery(self,'raw',"")
    
    def SetARDIDetails(self, url, assetid):
        """Set Connection Details

        This function sets some of the basic identity information about the core.

        Args:
            url: The URL to query to get core configuration information from ARDI
            assetid: The source ID that this core represents"""    
        self.url = url
        self.asset = assetid
        self.InitLogger()
    
    def SetARDIDriver(self, driver):
        """Set Driver

        This function associates the given driver with this core

        Args:
            driver: Associate this core with a driver"""       
        self.driver = driver
        self.driver.asset = self.asset
        driver.core = self
                 
    def Close(self):
		"""Closes to connection to the data store"""
		self.driver.Disconnect()

    def InitLogger(self):
        """Initialises the logging for this indiviual source"""

        if hasattr(self,'logger'):
            return
        
        #print 'Initialising Logger for ' + self.asset
        logfile = '/var/log/ardi/source-' + self.asset + '-' + self.profile + '-hist.log'
        if (platform.system() == 'Windows'):
                    logfile = "c:\\windows\\temp\\ardi-source-" + self.asset + '-' + self.profile + '-hist.log'

        ##The logger for a specific data source, rather than the driver process as a whole
        self.logger = logging.getLogger('Source '+self.asset)  
        if (len(self.logger.handlers) == 0):
            handler = logging.handlers.WatchedFileHandler(logfile)
            handler.setLevel(logging.INFO)
            formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            handler.setFormatter(formatter)
            self.logger.addHandler(handler)
            self.logger.info('----DRIVER STARTED----')
            self.logger.info('Starting Historian Data Source ' + self.asset + " on thread " + str(thread.get_ident()))
	
    def Reload(self):
        """Force the core to reload

        This function is usually called after a change to the required data"""
        self.LoadConfig()
        
    def LoadConfig(self):
        """Reload driver configuration

        This function triggers the driver to reload it's details from the ARDI server. Note that this
        is very different from the same function located in the DriverBase class. The data returned by
        this query includes all of the individual data points to be sampled from the data source."""
        if (self.source != ''):
			fl = open(self.source,'r')
			xml = fl.read()
			fl.close()
			self.parseconfig(xml)
			return
			
        response = StringIO()
        
        fullurl = "http://" + self.url + "/api/sourcedata?asset=" + str(self.asset) + "&mode=1"
        
        c = pycurl.Curl()
        c.setopt(c.URL,fullurl)
        c.setopt(c.WRITEFUNCTION,response.write)
        c.setopt(c.TIMEOUT,5)
        c.setopt(c.CONNECTTIMEOUT,5)
        c.perform()
        c.close()
        
        self.parseconfig(response.getvalue())
        
    def parseconfig(self,xml):
        """Parse the configuration details

        This function processes the XML from 'LoadConfig' to setup the driver and the many data points."""
        #print xml
        result = xmltodict.parse(xml)
        
        #print str(result)
        
        #Core Parameters
        self.driveraddr = result['sourcedata']['connection']['address']

        self.hostport = int(result['sourcedata']['connection']['port'])
        self.driver.SetAddress(self.driveraddr)

        if self.driveraddr is None:
            self.driveraddr = ""

        #Data Points
        #try:
        if True:
            self.pointlist = []
            self.transforms = {}
            if result['sourcedata']['points'] is not None:
                if '@address' in result['sourcedata']['points']['point']:
                    #There is a Single Point
                    
                    point = result['sourcedata']['points']['point']
                    if point['@address'] == '':
                        point['@address'] = str(self.dynamicaddress)
                        self.dynamicaddress += 1
                    point = result['sourcedata']['points']['point']
                    self.transforms[point['@address']] = point['@transform']
                    newpoint = pointinfo()
                    newpoint.address = point['@address']
                    newpoint.code = point['@code']
                    #newpoint.transform = point['@transform']
                    newpoint.changed = False
                    newpoint.value = ""
                    newpoint.intvalue = ""
                    self.pointlist.append(newpoint)
                else:
                    for point in result['sourcedata']['points']['point']:
                        #Add Data Points to List
                        if point['@address'] == '':
                            point['@address'] = point['@code']

                        self.transforms[point['@address']] = point['@transform']
                        #print "Saving Transform: " + point['@transform']
                        newpoint = pointinfo()
                        newpoint.address = point['@address']
                        newpoint.code = point['@code']
                        #newpoint.transform = point['@transform']
                        newpoint.changed = False
                        newpoint.value = ""
                        newpoint.intvalue = ""
                        self.pointlist.append(newpoint)
        #except:
        #    print "Failed To Load!";
        #    pass
        #print(str(self.transforms))
        self.logger.info(str(len(self.transforms)) + " Points/Transforms Loaded")
    
    def Query(self, points,start,end,function,grain,first):		
		"""Performs a query on a single source

        \param points: A list of points to query
        \param start: The start date for the query, in YYYY-MM-DD hh:mm:ss format
        \param end: The start date for the query, in YYYY-MM-DD hh:mm:ss format
        \param function: The function to perform. Functions include 'raw', 'avg', 'sum', 'min' and 'max'
        \param grain: The number of seconds between samples, or '0' for natural sampling rate
        \param first: True if this the data is to include the samples immediately preceeding the start time.
		"""
		
		#print "Starting From " + str(start)
		#print "Ending On " + str(end)

		finalpoints = []
		for pnt in points:
			for itm in self.pointlist:				
				if itm.code == pnt:
					finalpoints.append(itm.address)
					break
				
		if len(finalpoints) == 0:
			return "ERROR: No valid point specified"

		try:
		    return self.driver.Query(finalpoints,start,end,function,grain,first)
		except:
		    self.logger.exception("Historical Query Failed")
		    return None

    def GetResponder(self,func,endtime):
    	return historianquery(self,func,endtime)

    def AddRecord(self, id, stamp, value,addr):
        return self.qry.AddRecord(id,stamp,value,addr)

    def Launch(self,address,asset):
        """Starts the driver

        This function initiates the driver.

        Args:
            address: The encoded destination address (ie. IP address) of the device
            asset: The asset id that this source represents"""
        self.SetARDIDetails(address,asset)
        self.LoadConfig()    
        self.driver.Connect()
                    