## package ardi.driver.historianipc
#
#  Provides Historian Driver HTTP IPC
#
#  Classes in this namespace provide inter-process-communication between historian drivers
#  and ARDI through a TCP socket mirroring HTTP.
try:
	import SocketServer
except:
	import socketserver as SocketServer
	
try:
    from urllib.parse import parse_qsl
except:
    from urlparse import parse_qsl

from threading import Thread

## Function used to generate a query in a thread.
def QueryFunction(socket,inst,cmd):
    print("Query Thread Started")
    socket.Respond(inst,cmd)
    print("Query Thread Stopped")

## Historian IPC Message Handler
#
#    This class responds to commands that are issued from server applications such as ARDI.
#
#    Some commands operate on the driver as a whole, while others operate on a single core
class IPCHandler(SocketServer.StreamRequestHandler):

    ## Handles Incoming Commands
    #
    #    This function reads from the incoming stream and response appropriately."""
    def handle(self):
        self.data = self.rfile.readline().strip()
        cmd = self.data.split('|')
        context = 1
        if len(cmd)>2:
            context = cmd[2]
            
        #print "Incoming Command: " + cmd[0]  
        try:            
            if cmd[0] == 'spinup':
                if len(cmd) > 1:
                    self.server.core.Spinup(cmd[1],context)
                    return
            if cmd[0] == 'reload':
                if len(cmd) > 1:
                    self.server.core.Reload(cmd[1],context)
                    return
            if cmd[0] == 'spindown':
                self.server.core.Restart()
                return
            if cmd[0] == 'restart':
                self.server.core.Restart()
                return
            if cmd[0] == 'query':
                #Rebuild query...
                str = ""
                for x in range(2,len(cmd)):
                    if x > 2:
                        str = str + "|"
                    str = cmd[x]
                    
                self.ThreadRespond(cmd[1],str)
                return
        except:
            pass

        if self.data[0:14] == "GET /query?id=":
            #Get the ID number...
            pos = self.data.find("&")
            if pos > 0:
                id = self.data[14:pos]            
                self.ThreadRespond(id,self.data[pos+1:len(self.data)-9])

    ## Threaded response handler
    def ThreadRespond(self,instance,cmd):
        self.Respond(instance,cmd)
        #reactor.callInThread(QueryFunction,self,instance,cmd)

    ## Sends a query response to the client
    #
    #    This function issues a query via the driver and returns the results.
    #   
    #    \param instance The ARDI source asset ID
    #    \param cmd The query command string"""
    def Respond(self,instance,cmd):
        
        #Split Query String

        vals = parse_qsl(cmd)
        
        points = ""
        start = ""
        end = ""
        func = "raw"
        grain = 0
        first = 1
        context = 1
        
        params = cmd.split("&")
        for x in vals:            
            if x[0] == "points":
                points = x[1].split("_")
            if x[0] == "start":
                start = x[1]
            if x[0] == "end":
                end = x[1]
            if x[0] == "func":
                func = x[1]
            if x[0] == "function":
                func = x[1]
            if x[0] == "grain":
                grain = x[1]
            if x[0] == "followup":
                first = 0
            if x[0] == "context":
                context = x[1]

        #print "Requesting History..."
        result = self.server.core.Query(instance,points,start,end,func,grain,first,context)
        #print "Sending Response: " + result
        self.wfile.write(result)

## A Twisted mixin, creating a threaded TCP server.
class ThreadedTCPServer(SocketServer.ThreadingMixIn, SocketServer.TCPServer):
    pass

## Historical IPC (Socket) server to control a driver
#
#    This class listens for socket connections and responds to commands from controlling applications (such as ARDI)
#
#    Some commands operate on the driver as a whole, while others operate on a single core
class DriverIPC:
    def __init__(self):
        """Intialises the IPC Connection"""

        ##The TCPServer instance that represents this service
        self.service = None

        ##The port that this IPC service should run on
        self.port = 9001

        ##A reference to the drivercore that spawned this IPC service
        self.core = None        

    def start(self):
        """Start the IPC Service

        This command starts the TCPServer that listens for incoming commands"""
        SocketServer.TCPServer.allow_reuse_address = 1
        self.service = SocketServer.TCPServer(("",self.port),IPCHandler)
        #self.service = ThreadedTCPServer(("",self.port),IPCHandler)
        self.service.core = self.core
        self.service.serve_forever()    

#IPC = DriverIPC()
#IPC.start()
