## @package ardi.driver.transform
#  Transforms data based on ARDI transform syntax.
#
import sys
import traceback

## Converts Text
#
#    This function is responsible for converting data to a more human-readable form. This includes a number of data transformations including scaling, inversion
#    and some basic math.
class transform:

    ## Initialises the Transform
    def __init__(self,st):

        ##Contains the body of the format string
        self.st = st
        self.lines = []
        self.parse(st)        

    ## Loads the transformation format
    def parse(self,st):
        self.st = st

        ##A split version of the format string with various parameters in different indexes
        self.lines = st.split('^')
        #self.bits = st.split('|')        

    ## Applies a sign fix to unsigned short integers
    #
    #    This function is for a special case often associated with recording raw Modbus values incorrectly.
    #    It converts a number from an unsigned short to an signed one
    #
    #    This was added as a feature for South32 - Joy Mining systems stored some signed
    #    values as unsigned values, resulting in some very awkward results.
    #
    #     @param value The incoming value
    #     @param mult The multiplier that should be applied to the value before conversion (ie. 10 to add a decimal point to the last place of the number)"""
    def signfix(self,value,mult):
        
        m = float(mult)        
        v = int(float(value) * m)        
        if v < 0:            
            pass
        else:            
            if v > 32000:                                
                v = v ^ 32768
                v = 32768 - v
                v = -v
            pass
        v = float(v) / m        
        return v

    ## Adds or subtracts a number to the amount.
    #
    #  @param value The current value
    #  @param amount The amount to offset the value by. Negative values subtract.
    def offset(self, value, amount):
        am = float(amount)
        val = float(value)
        val += am
        return val

    ## Reduces the value to zero if it is below a given threshold.
    #
    #  @param value The current value
    #  @param amount The threshold amount. 
    def lowcut(self, value, amount):
        am = float(amount)
        val = float(value)
        if val < am:
            val = 0
        return val

    ## Compares one number to another and reduces the output to a binary 1/0.
    #
    #  @param value The current value
    #  @param amount The threshold amount
    #  @param direction 'g' to output '1' for high values, or 'l' to output '1' for low values.
    def boolpoint(self, value, amount,direction):
        am = float(amount)
        val = float(value)
        if direction == 'g':
            if val >= am:
                return '1'
            else:
                return '0'
        else:
            if val <= am:
                return '1'
            else:
                return '0'
        return 0

    ## Rounds the number to the given number of decimal places
    #
    #  @param value The current value
    #  @param points The number of decimal places
    def rounding(self, value, points):        
        am = int(points)
        val = float(value)
        if (am == 0):
            return int(float(value))
        else:
            return round(val,am)

    ## Re-scale a number.
    #
    #  Note that this can also be used to re-range or invert a value.
    #
    #  @param value The current value
    #  @param minin The minimum input value.
    #  @param minout The minimum output value
    #  @param maxin The maximum input value
    #  @param maxout The maximum output value
    def scale(self, value, minin,maxin, minout, maxout):
        inmin = float(self.bits[1])
        inmax = float(self.bits[2])
        outmin = float(self.bits[3])
        outmax = float(self.bits[4])
        if (isinstance(value,basestring)):
            perc = (float(value.decode('ascii'))-inmin) / (inmax - inmin)
        else:
            perc = (value-inmin) / (inmax - inmin)
        return (perc * (outmax - outmin)) + outmin

    ## Convert values
    #
    #  This applies a match operation - outputing value A where value B is present.
    def match(self, value):        
        
        for i in xrange(1,len(self.bits)-1,2):
            #print '  checking for "' + str(self.bits[i]) + '" (' + str(value) + ')'
            if str(value) == str(self.bits[i]):
                return self.bits[i+1]
        
        return '-1'

    ## Clamps value to range
    #
    #  Forces the value to not leave the specified range.
    def clamp(self, value, mini, maxi):        
        if float(value) < float(mini):
            return float(mini)
        if float(value) > float(maxi):
            return float(maxi)

        return value

    ## Changes to 'error' if outside range
    #
    #  Returns '^' if the value is outside the specified range.    
    def vrange(self, value, mini, maxi):
        #print("Testing " + str(value))
        if float(value) < float(mini):
            print(str(value) + " is under threshold " + str(mini))
            return '^'
        if float(value) > float(maxi):
            print(str(value) + " is over threshold " + str(maxi))
            return '^'

        return value

    ## Doesn't report values if outside range.
    #
    #  Returns '^^' (ie. don't process) if the value is outside the specified range.    
    def holdbad(self, value):
        if value == "^":
            return "^^"

    ## Changes to 'error' if an exact match with a value
    #
    #  Returns '^' if the value is exactly equal to a given number    
    def badat(self, value,badvalue):        
        if value == mini:
            value = '^'        

        return value        

    ## Applies the transformation
    #
    #  @param value The value to be transformed.
    def run(self,value):

        #print "Running"
        #print self.lines

        v = value
        for ln in self.lines:
            v = self.runline(ln,v)

        return v

    ## Applies a single line of the transformation
    #
    #There are a number of possible transforms, based on the string used in "Parse"
    #
    #SCALE|&lt;in-min&gt;|&lt;in-max&gt;|&lt;out-min&gt;|&lt;out-max&gt;
    #
    #    Scales the number from the 'in' minimums and maximums, to the 'out' minimums and maximums. This can also be used to
    #    invert a number.
    #
    #BINARY|&lt;match&gt;|&lt;val&gt;|&lt;match&gt;|&lt;val&gt;
    #MATCH|&lt;match&gt;|&lt;val&gt;|&lt;match&gt;|&lt;val&gt;
    #
    #    If the value equals 'match', output 'val'
    #
    #SIGNFIX
    #
    #
    #    Apply a short-integer sign fix to the data (Added for Joy Mining databases that store data into their DB incorrectly)
    #
    #LOWCUT|&lt;point&gt;
    #
    #    Ignore values under a certain threshold
    #
    #BPOINT|&lt;point&gt;|&lt;dir&gt;
    #
    #    Translate to a boolean value based on a specific point
    #
    #OFFSET|&lt;point&gt;
    #
    #    Add/subtract a certain offset (zero)
    #
    #ROUND|&lt;points&gt;
    #
    #    Round off to a given number of decimal points.
    #
    # @param value: The value to be transformed
    def runline(self,line,value):
        
        
        self.bits = line.split('|')

        #print self.bits
        
        try:
            if self.bits[0] == 'SCALE' or self.bits[0] == 'SCALING':
                return self.scale(value,self.bits[1],self.bits[2],self.bits[3],self.bits[4])
            
            if self.bits[0] == 'BINARY':
                return self.match(value)
            
            if self.bits[0] == 'MATCH':
                return self.match(value)
            
            if self.bits[0] == 'OFFSET':                
                return self.offset(value,self.bits[1])
            
            if self.bits[0] == 'LOWCUT':                
                return self.lowcut(value,self.bits[1])

            if self.bits[0] == 'BPOINT':
                bpd = 'g'
                if len(self.bits) > 2:
                    bpd = self.bits[2]
                return self.boolpoint(value,self.bits[1],bpd)
            
            if self.bits[0] == 'ROUND':                
                return self.rounding(value,self.bits[1])
            
            if self.bits[0] == 'SIGNFIX':                
                return self.signfix(value,float(self.bits[1]))

            if self.bits[0] == 'CLAMP':
                return self.clamp(value,float(self.bits[1]),float(self.bits[2]))

            if self.bits[0] == 'RANGE':
                return self.vrange(value,float(self.bits[1]),float(self.bits[2]))
            
            if self.bits[0] == 'BADAT':
                return self.badat(value,float(self.bits[1]))

            if self.bits[0] == 'HOLDBAD':
                return self.holdbad(value)

        except:
            #print("Exception Processing Transform '"+ str(line)+ "' on '"+str(value)+"'")
            #traceback.print_exc()
            return "^"
        
        return value
