## @package ardi.driver.eventdriverbase
#  ARDI Event Driver Service
#
#  The 'Base' class encapsulates all of the essential startup and control behaviours of an ARDI event driver.
#
#  Each 'Base' contains multiple 'Core' objects, which in turn contain a single source instance.
#
import os
import platform
import time
import sys
import logging
import logging.handlers
import threading
import ardi.driver.eventipc as eventipc
import ardi.driver.eventdrivercore as eventdrivercore
import ardi.driver.logsystem as logsystem
import argparse
#import pycurl
import xmltodict
import traceback
import requests
#from StringIO import StringIO
from twisted.web import server,http,resource
import json

## Structure for source details
#
#  This class is used as a structure to hold details about a single ARDI data source.
class ardisourcedetail:
    def __init__(self):
        self.id = 0
        self.subport = 5336
        self.subhost = "localhost"
        self.mode = 0
        self.profile = 1

def corethread(core, servername, assetid):
    core.Launch(servername, assetid)

## Starts the webservice used to communicate with and manage this driver
#
# \param core: The base driver that is controlled by this web service
def webservice(core):
    
    core.logger.info('Starting IPC Service')    
    
    ipc = eventipc.DriverIPC()
    ipc.core = core
    ipc.port = core.hostport
    if ipc.port == 0:
        ipc.port = 8081
        
    try:
        ipc.start()
    except KeyboardInterrupt:
        pass


## The base class for an event driver.
#
#    This class is responsible for all stages of driver life-cycle.
#
#    It also communicates directly with ARDI to get details on what data sources it should serve via HTTP web requests
class eventdriver:

    ## Initialises the event driver
    def __init__(self):
        ##The IPC port for this driver
        self.hostport = 10000
        self.assets = []
        self.basiclog = []
        self.detaillog = []
        
        pass
    
    ## Prints usage information if startup fails
    def usage(self):
        print("Event Driver Startup: <drivername> <asset1>,<asset2>...<assetn>")
        return

    ## Initialises the event logger
    def initlogger(self):

        fname = "/var/log/ardi/driver-" + str(self.hostport) + ".log"
        if (platform.system() == 'Windows'):
            try:
                    logfolder = os.environ['ARDILogPath']
            except:
                    logfolder = "c:\\windows\\temp"
            fname = logfolder + "\\ardi-driver-" + str(self.hostport) + ".log"
        
        ##The logger used to record information about the driver process as a whole
        self.logger = logsystem.CreateLogger(self,"Driver " + str(self.hostport),fname)
        self.logger.propagate = False

        self.logger.info('----EVENT DRIVER RESTART----')
        self.logger.info('Starting Event Driver Process')
        
        redir = logsystem.iologger(self.logger,logging.ERROR)
        sys.stderr = redir

    ## Start the Driver
    #
    #    This function is called once all initialisation is complete. It launches sources and the IPC service.
    #
    #   \param driverfactory: The factory class that produces driver instances
    def start(self,driverfactory):
        

        ##A class that generates event drivers on request
        self.factory = driverfactory

        arglist = sys.argv
        if len(arglist) >= 2:
            if arglist[1] == 'start':                
                arglist = arglist[2:]
            else:
                arglist = arglist[1:]
  
        parser = argparse.ArgumentParser(description='ARDI Driver')
        parser.add_argument('driverid',metavar='PortNo',type=int, help='Unique ID/port for the driver', nargs='?')
        parser.add_argument('server',metavar='Server', help='Server Name', nargs='?', default='localhost')
        parser.add_argument('assetid',metavar='AssetNo',type=int, help='Asset IDs to act as a driver for', nargs='?')
        parser.add_argument('--multiple',dest='multi',action='store_const',const=True,default=False,help='Allows this driver to connect to multiple devices')        
        parser.add_argument('--script','-r',dest='script',default='')
        parser.add_argument('--shell','-l',dest='shell',default='')
        parser.add_argument('--config','-c',dest='configfile',default='')        
        parser.add_argument('--debuglevel',dest='debuglevel',default=2)
        args = parser.parse_args(arglist)
        
        #self.logger.debug(args)        
        
        servername = args.server
        self.hostport = int(args.driverid)

        self.initlogger()

        self.servername = args.server
        self.args = args
        
        self.logger.info('Driver ' + str(self.hostport) + " feeds ARDI server " + servername)

        #self.script = args.script
        #self.shell = args.shell        

        ##A list of the currently active ARDI sources
        self.drivers = {}

        ##A quick lookup dictionary to map addresses to ARDI names
        self.assetids = []
        if args.assetid != None:
            for assetid in args.assetid:
                self.assetids.append(assetid)
        else:
            while True:
                try:
                    #response = StringIO()
                
                    fullurl = "http://" + servername + "/api/driverdata?port=" + str(self.hostport) + "&mode=4"                    
                                    
                    web = requests.get(fullurl,timeout=10)
                    break
                except:
                    time.sleep(10)
                    pass
            #print response.getvalue()

            self.parseconfig(web.text)
            
            self.logger.info( "Load Default Configuration From Server")

        #for assetid in self.assetids:
        for asst in self.assets:
            assetid = asst.id
            self.logger.info("Creating event driver instance for " + str(assetid))
            try:
                instance = driverfactory.createinstance()
                core = eventdrivercore.eventcore(instance)
                core.base = self
                core.source = args.configfile
                core.profile = asst.profile
                self.drivers[str(assetid) + "_" + str(core.profile)] = core
            except:
                traceback.print_exc()
            try:
                core.Launch(servername,assetid)
            except:
                print("Launch Failure")
                traceback.print_exc()

        #print self.drivers
        print("Time to start web service...")
        webservice(self)

    ## Imports the configuration XML file
    #
    #    This function loads the list of data points within the XML text provided. The function is usually called from LoadConfig
    #
    #    \param xml: The configuration data in XML format"""
    def parseconfig(self,xml):
        result = xmltodict.parse(xml)
                        
        #Core Parameters
        try:            
            if '@id' in result['driverdata']['source']:
                ast = ardisourcedetail()
                ast.id = result['driverdata']['source']['@id']
                ast.profile = result['driverdata']['source']['@profile']
                ast.mode = result['driverdata']['source']['@mode']
                self.assets.append(ast)
                
                self.assetids.append(result['driverdata']['source']['@id'])
            else:
                for itm in result['driverdata']['source']:
                    ast = ardisourcedetail()
                    ast.id = itm['@id']
                    ast.profile = itm['@profile']
                    ast.mode = itm['@mode']
                    self.assets.append(ast)
                
                    self.assetids.append(itm['@id'])
        except:
            pass
            
        self.lookups = {}
        #print(str(result))
        try:
            if result['driverdata']['lookups'] == 'present':
                self.loadLookups()
        except:
            pass

    ## Performs a query on a single source
    #
    #    \param assetid: The SOURCE asset id
    #    \param points: A list of points to query
    #    \param start: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param end: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param function: The function to perform. Functions include 'raw', 'avg', 'sum', 'min' and 'max'
    #    \param grain: The number of seconds between samples, or '0' for natural sampling rate
    #    \param first: True if this the data is to include the samples immediately preceeding the start time.
    def Query(self,instance,context,params):
        #print("Querying Context: " + str(instance) + '_' + str(context))        
        return self.drivers[str(instance) + '_' + str(context)].Query(instance,params)

    ## Forces an event source to reload
    #
    #    This function is usually called internally by a change to the ARDI data.
    #
    #    \param assetid: The SOURCE asset id            
    def Reload(self, assetid,context=1):
        self.drivers[str(assetid) + "_" + str(context)].Reload()

    def ShowStatus(self):
        final = []        
        for core in self.drivers:            
            try:
                final.append(self.drivers[core].Status())
            except:
                traceback.print_exc()        
        return json.dumps(final)


    def ShowLog(self,coreid,context,logtype):        
        if coreid == 0:
            if logtype == "basic":
                return "\n".join(self.basiclog)                
            else:
                return "\n".join(self.detaillog)

        #print("Checking for ID: " + str(coreid))
        
        for core in self.drivers:            
            cr = self.drivers[core]            
            if int(cr.asset)==int(coreid):                
                if int(cr.profile)==int(context):
                    if logtype == "basic":                
                        return "\n".join(cr.basiclog)
                    else:
                        return "\n".join(cr.detaillog)
        return ""

    def ShowIssues(self,coreid,context):
        if coreid == 0:
            return ""
        for core in self.drivers:
            cr = self.drivers[core]            
            if int(cr.asset)==int(coreid):
                if int(cr.profile)==int(context):
                    try:
                        return "\n".join(cr.driver.issues)
                    except:
                        return "Driver Does Not Support 'Issues'"
        return ""

    ## Setup a new event source
    #
    #    This function is usually called internally by a change to the ARDI data.
    #
    #    \param assetid: The SOURCE asset id    
    def Spinup(self,assetid,context=1):
        try:
            self.drivers[str(assetid) + "_" + str(context)].Reload()
            self.logger.info("Reloaded event source " + str(assetid) + " due to configuration change")
        except:
            self.logger.info("Spinning up event driver instance for " + str(assetid))

            try:                
                fullurl = "http://" + self.servername + "/api/driverdata?port=" + str(self.hostport) + "&mode=4"                
                web = requests.get(fullurl,timeout=10)                
            except:
                self.logger.info("Driver information not available")
                return

            self.parseconfig(web.text)

            for asst in self.assets:                
                if asst.id == assetid:

                    try:
                        instance = self.factory.createinstance()
                        core = eventdrivercore.eventcore(instance)
                        core.base = self
                        core.source = self.args.configfile
                        core.profile = asst.profile
                        self.drivers[str(assetid) + "_" + str(asst.profile)] = core
                        self.logger.info("Launching connection/source " + str(assetid))
                    except:
                        traceback.print_exc()

                    try:                        
                        core.Launch(self.servername,assetid)
                        return
                    except:
                        self.logger.error("Launch Failed")
                        traceback.print_exc()
                        pass
            self.logger.info("Can't find asset " + str(assetid))


    ## Forces the entire driver process to reload
    #
    #   @todo Note that this is currently unimplemented for event drivers!
    #
    def Restart(self):
        
        self.logger.info('Driver Restart Requested via IPC')
        #self.StopDrivers()
        #self.LoadConfig()
