import os
import sys
import time
import errno
import logging
from logging.handlers import BaseRotatingHandler
from stat import ST_DEV, ST_INO

try:
    import win32file
    import msvcrt
except:
    pass

def OpenUnlockedFile(filename,writing=None):
    if 'win32file' in sys.modules:
        #Try opening the file with Win32.CreateFile so it can be modified during access.
        h = win32file.CreateFile(filename,win32file.GENERIC_READ|win32file.GENERIC_WRITE,win32file.FILE_SHARE_DELETE | win32file.FILE_SHARE_READ,None,win32file.OPEN_ALWAYS,0,None)
        handle = h.Detach()
        descript = msvcrt.open_osfhandle(handle,os.O_RDWR)
        return open(descript,"w+")
    else:
        #Just use the old-fashioned file open method
        return open(filename,"w+")


class WatchedRotatingFileHandler(logging.FileHandler):
    """
    Combination of RotatingFileHandler and WatchedFileHandler.
    """

    def __init__(self, filename, mode="a", maxBytes=0, backupCount=0, encoding=None):
        # RotatingFileHandler
        if maxBytes > 0:
            mode = "a" # doesn"t make sense otherwise!

        #if codecs is None:
        #    encoding = None
        logging.FileHandler.__init__(self, filename, mode, encoding)
        self.mode = mode
        self.encoding = encoding
        self.maxBytes = maxBytes
        self.backupCount = backupCount

        if self.stream is not None:
            try:
                self.stream.flush()
            except:
                pass
            self.stream.close()
            self.stream = OpenUnlockedFile(self.baseFilename,"w")

        # WatchedFileHandler
        if not os.path.exists(self.baseFilename):
            self.dev, self.ino = -1, -1
        else:
            stat = os.stat(self.baseFilename)
            self.dev, self.ino = stat[ST_DEV], stat[ST_INO]

    def check_stat(self):   
        """
        Check if log file changed.
        """         
        stat = None
        changed = 1

        if os.path.exists(self.baseFilename):
            try:
                stat = os.stat(self.baseFilename)
                changed = (stat[ST_DEV] != self.dev) or (stat[ST_INO] != self.ino)
            except OSError:
                pass
        return stat, changed

    def emit(self, record):
        """
        Emit a record.

        First check if the underlying file has changed, and if it
        has, close the old stream and reopen the file to get the
        current stream.

        Output the record to the file, catering for rollover as described
        in doRollover().
        """
        # WatchedFileHandler
        stat, changed = self.check_stat()

        if changed and self.stream is not None:
            try:
                try:
                    self.stream.flush()
                except:
                    pass
                self.stream.close()
                self.stream = OpenUnlockedFile(self.baseFilename,"w")
                if stat is None:
                    try:
                        stat = os.stat(self.baseFilename)
                    except OSError:
                        return
                self.dev, self.ino = stat[ST_DEV], stat[ST_INO]
            except ValueError:
                pass

        # RotatingFileHandler
        try:
            if self.shouldRollover(record):
                with FileLock(self.baseFilename, timeout=1, delay=0.1, lock_file_contents=os.getpid()):
                    # check again and do rollover if necessary
                    if self.shouldRollover(record):
                        self.doRollover()
            logging.FileHandler.emit(self, record)
            #print(str(record))
        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)

    def doRollover(self):
        """
        Do a rollover.
        """        
        stat, changed = self.check_stat()
        if changed:
            return

        print("Performing Log Rollover For " + self.baseFilename + "...")

        try:
            try:
                self.stream.flush()
            except:
                pass

            self.stream.close()
            time.sleep(0.5)
            if self.backupCount > 0:
                for i in range(self.backupCount - 1, 0, -1):
                    sfn = "%s.%d" % (self.baseFilename, i)
                    dfn = "%s.%d" % (self.baseFilename, i + 1)
                    if os.path.exists(sfn):
                        if os.path.exists(dfn):
                            os.remove(dfn)
                        os.rename(sfn, dfn)
                dfn = self.baseFilename + ".1"
                if os.path.exists(dfn):
                    os.remove(dfn)
                os.rename(self.baseFilename, dfn)
            
                self.stream = OpenUnlockedFile(self.baseFilename, "w")
        except:
            return

    def shouldRollover(self, record):
        """
        Determine if rollover should occur.

        Basically, see if the supplied record would cause the file to exceed
        the size limit we have.
        """
        if self.maxBytes > 0:                   # are we rolling over?
            msg = "%s\n" % self.format(record)
            try:
                self.stream.seek(0, 2)  #due to non-posix-compliant Windows feature
                if self.stream.tell() + len(msg) >= self.maxBytes:
                    return 1
            except:
                return 0
        return 0


class FileLock(object):
    """
    A file locking mechanism that has context-manager support so 
    you can use it in a ``with`` statement. This should be relatively cross
    compatible as it doesn"t rely on ``msvcrt`` or ``fcntl`` for the locking.
    
    Adapted from:
    http://www.evanfosmark.com/2009/01/cross-platform-file-locking-support-in-python/
    """
 
    class FileLockException(Exception):
        pass
 
    def __init__(self, protected_file_path, timeout=1, delay=0.1, lock_file_contents=""):
        """
        Prepare the file locker. Specify the file to lock and optionally
        the maximum timeout and the delay between each attempt to lock.
        """
        self.lockfile = protected_file_path + ".lock"
        self.timeout = timeout
        self.delay = delay
        self.lock_file_contents = lock_file_contents
        self.lock_fd = None

        self.is_locked = False
 
    def acquire(self, blocking=True):
        """ 
        Acquire the lock, if possible. 
        If the lock is in use, and `blocking` is False, return False.
        Otherwise, check again every `self.delay` seconds until it either gets 
        the lock or exceeds `timeout` number of seconds, in which case it 
        raises an timeout exception.
        """
        start_time = time.time()
        if self.lock_fd is not None:
            try:
                os.close(self.lock_fd)
            except:
                pass
            self.lock_fd = None

        while True:
            try:
                # Attempt to create the lockfile.
                # These flags cause os.open to raise an OSError if the file already exists.
                self.lock_fd = os.open( self.lockfile, os.O_CREAT | os.O_EXCL | os.O_RDWR )
                break
            except OSError as e:
                if e.errno != errno.EEXIST:
                    raise 
                if (time.time() - start_time) >= self.timeout:
                    raise FileLock.FileLockException("Timeout occurred.")
                if not blocking:
                    return False
                time.sleep(self.delay)
        self.is_locked = True
        return True
 
    def release(self):
        """ 
        Get rid of the lock by deleting the lockfile. 
        When working in a `with` statement, this gets automatically 
        called at the end.
        """
        if self.is_locked:
            if self.lock_fd is not None:
                try:
                    os.close(self.lock_fd)
                except:
                    pass
            self.lock_fd = None

            self.is_locked = False
            os.unlink(self.lockfile)
 
    def __enter__(self):
        """ 
        Activated when used in the with statement. 
        Should automatically acquire a lock to be used in the with block.
        Timeout means the lock is dummy, so it will remove dummy lock and 
        acquire the lock anyway.
        This is not a strict lock, just try best to keep synced.
        """
        try:
            self.acquire()
        except FileLock.FileLockException:
            os.unlink(self.lockfile)
            self.acquire()
        return self
 
    def __exit__(self, type, value, traceback):
        """ 
        Activated at the end of the with statement.
        It automatically releases the lock if it isn"t locked.
        """
        if self.is_locked:
            self.release()
 
    def __del__(self):
        """ 
        Make sure this ``FileLock`` instance doesn"t leave a .lock file
        lying around.
        """
        if self.is_locked:
            self.release()

class iologger:
    def __init__(self,logger,level):
        self.logger = logger
        self.linebuf = ''
        self.log_level = level
        
    def write(self,buf):
        for line in buf.rstrip().splitlines():
            self.logger.log(self.log_level,line.rstrip())

    def flush(self):
        try:
            self.logger.flush()
        except:
            pass

#Takes in basic (ie. error and warning) messages, recording to an internal buffer.
class LocalBufferHandler(logging.StreamHandler):
    def __init__(self,store):
        logging.StreamHandler.__init__(self)
        self.store = store

    def emit(self,record):        
        ln = self.format(record)
        self.store.basiclog.append(ln)
        print(ln)
        if len(self.store.basiclog) > 200:
            del self.store.basiclog[0]

#Takes in detailed (ie. info and debug) messages, recording to an internal buffer
class LocalDetailBufferHandler(logging.StreamHandler):
    def __init__(self,store):
        logging.StreamHandler.__init__(self)
        self.store = store

    def emit(self,record):
        self.store.detaillog.append(self.format(record))
        if len(self.store.detaillog) > 1000:
            del self.store.detaillog[0]

def CreateLogger(host,loggername,filename):
   
    print("Setting Up Logging: " + loggername + " @ " + filename)
    rootlogger = logging.getLogger()
    rootlogger.setLevel(logging.INFO)

    ##The log file that represents the driver process as a whole
    logger = logging.getLogger(loggername)
    logger.propagate = False                      
            
    #File-based IO        
    loghandler = WatchedRotatingFileHandler(filename,mode="a",maxBytes=1048576,backupCount=3)
    loghandler.setLevel(logging.INFO)

    #Memory buffer for major events
    localservehandler = LocalBufferHandler(host)
    localservehandler.setLevel(logging.INFO)

    #Memory buffer for minor events
    localdetailhandler = LocalDetailBufferHandler(host)
    localdetailhandler.setLevel(logging.DEBUG)
            
    # create a logging format            
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    loghandler.setFormatter(formatter)
    localservehandler.setFormatter(formatter)
    localdetailhandler.setFormatter(formatter)
            
    # add the handlers to the logger            
    logger.addHandler(loghandler)
    logger.addHandler(localservehandler)
    logger.addHandler(localdetailhandler)    

    #Redirect errors
    #redir = iologger(logger,logging.ERROR)
    #sys.stderr = redir

    return logger