import json
import traceback
import importlib.util
from twisted.web import server, resource
from twisted.internet import reactor
import datetime
import pytz
import traceback
import urllib
import threading
import re
import sys
import os
import logging

def validOption(condition,options,default=False):
        
        if condition == "":
            return default 
        
        try:
            if condition in options:     
                return True        
        except:
            pass
        
        expressions = re.findall("([a-zA-Z0-9_]*)([<=>]*)([a-zA-Z0-9_]*)",condition)        
        if expressions == None or expressions[0][0] == "":           
            if condition in options:     
                return True
        else:
            for results in expressions:         
                if results[1] == "":
                    continue                
                if results[0] in options:
                    if results[1] == '':
                        return True

                    if results[1] == '=':                        
                        if str(options[results[0]]) == str(results[2]):                            
                            return True
                    if results[1] == '!=':
                        if str(options[results[0]]) != str(results[2]):
                            return True
                    if results[1] == '>=':
                        if str(options[results[0]]) >= str(results[2]):
                            return True
                    if results[1] == '<=':
                        if str(options[results[0]]) <= str(results[2]):
                            return True
                    if results[1] == '>':
                        if str(options[results[0]]) > str(results[2]):
                            return True
                    if results[1] == '<':
                        if str(options[results[0]]) < str(results[2]):
                            return True
        
        return False

def checkFilters(d,name,options):
    #print("Filtering :" + name)
    #print("Filer: " + str(d.filter))
    #print("Allow: " + str(d.allow))
    #print("Deny: " + str(d.deny))
    #print("Options: " + str(options))
    proceed = True    
    if d.filter is not None:                         
        if re.search(d.filter,name) is None:             
            return False

    if d.allow is not None:                  
        for q in d.allow:         
            #print("Checking For Option " + q + " in " + str(options))
            if not validOption(q,options):     
                proceed = False
                break

    if d.deny is not None:
        for q in d.deny:
            if validOption(q,options):
                proceed = False
                break    

    return proceed

#Defines a key-value output path
class KVOutput:
    def __init__(self,name,configfile="output.json",modules="",ignore=[]):
        self.name = name
        self.configfile = configfile
        self.destinations = []
        self.webcontent = {}
        self.buffer = []
        self.webhost = None
        self.modules = modules        
        if modules == "":
            for q in sys.path:
                final = q + "/ardi/util/outputengine"                
                if os.path.exists(final):
                    modules = final
                    self.modules = final
                    break                                
        else:
            self.modules = modules

        self.values = {}
        self.ignore = ignore
        self.initialised = False
        if self.modules != "":
            self.modules += "/"
        self.mods = {}
        self.channels = {}
        self.logger = logging.getLogger(__name__)
        self.initialising = False
        #self.logger.setLevel(logging.DEBUG)
        self.LoadAll(configfile)    

    def LoadAll(self,configfile):
        
        self.logger.info("Loading MOS Configurations From " + str(configfile))
        if configfile[0] == '#':
            self.Load(configfile)
            return

        filelist = []
        if os.path.isfile(configfile):
            filelist.append(configfile)

        fname = os.path.basename(configfile)
        lastpath = os.path.dirname(configfile)
        pth = os.path.dirname(os.path.dirname(configfile))
        while pth != lastpath:            
            p = pth + "/" + fname
            if os.path.isfile(p):
                filelist.append(p)
            lastpath = pth
            pth = os.path.dirname(pth)

        #print("Using Config Files: " + str(filelist))
        for f in filelist:
            self.Load(f)

    #Load the output configuration
    def Load(self,configfile):
        self.logger.info("Loading Individual MOS Configurations From " + str(configfile))
        try:
            if configfile[0] == '[':
                jsn = json.loads(configfile)
            else:
                fl = open(configfile,'r')
                content = fl.read()
                jsn = json.loads(content)
                fl.close()

            for x in jsn:
                if x['method'] in self.ignore:
                    print("This application is not compatible with " + x['method'] + " output.")
                    continue
                if 'type' in x:
                    if x['type'].lower() != 'value':
                        continue

                self.AddDestination(x)
                
        except:
            traceback.print_exc()
            print("No Output Configuration File Found")
            pass

    #Load a destination module
    def LoadDestination(self,d):
        method = d['method']
        
        #Try and calculate the final name for this...
        ob = None
        nm = ""
        if 'name' in d:
            nm = d['name']
        else:
            nm = method
            for k in d:
                if k == 'method':
                    continue
                if nm != "":
                    nm += ":"
                nm += str(d[k])

        if method not in self.mods:
            library = 'kv_' + method
            try:
                spec = importlib.util.spec_from_file_location(library,self.modules + library+".py")
                module = importlib.util.module_from_spec(spec)
                spec.loader.exec_module(module)

                self.mods[method] = module
            except Exception as e:
                self.logger.warn("Can't Load " + method + " module - " + str(e))
                traceback.print_exc()                     

            try:
                ob = module.output(d)
                ob.module = method
                ob.config = d
                ob.cname = self.GetName(ob)                
            except:
                traceback.print_exc()
                #print(str(dir(module)))
                return None
        else:
            ob = self.mods[method].output(d)
            ob.module = method
            ob.config = d
            ob.cname = self.GetName(ob)
    
        if ob is not None:

                ob.webendpoint = None
                try:            
                    ob.hierarchy = ob.Supports('hierarchy')
                except:
                    ob.hierarchy = False
                    pass
                
                try:            
                    ob.webendpoint = ob.Supports('html')
                    if ob.webendpoint == False:
                        ob.webendpoint = None

                    if ob.webendpoint is not None:                    
                        if 'port' in d:
                            ob.webport = d['port']
                        else:
                            ob.webport = 8199
                except:
                    ob.webendpoint = None
                    pass

                try:
                    ob.filter = d['filter']
                except:
                    ob.filter = None

                try:
                    ob.allow = d['allow']
                    if isinstance(d['allow'],str):
                        tags = d['allow'].split(' ')
                        ob.allow = tags
                except:
                    ob.allow = None

                try:
                    ob.deny = d['deny']
                    if isinstance(d['deny'],str):
                        tags = d['deny'].split(' ')
                        ob.deny = tags
                except:
                    ob.deny = None

                ob.name = self.name + " " + d['method'].upper() + " Output"

        return ob

    #Add a destination
    def AddDestination(self,d):
        self.logger.debug("Adding Destination: " + str(d))
        try:
            ob = self.LoadDestination(d)
            if ob is not None:
                self.logger.debug("Loaded Destination: " + ob.cname)

                self.destinations.append(ob)
        except Exception as e:
            #print("Failed to load destination: " + d['method'])
            self.logger.warn("Unable to add destination: " + str(e))
            traceback.print_exc()
            return False
        return True

    #Initialise the system (connect to any data stores)
    def Initialise(self):                
        self.logger.info("Initialising Key/Value MOS Output for " + str(len(self.destinations)) + " destinations.")
        if self.initialised == True or self.initialising == True:            
            return
        
        self.initialising = True        

        for d in self.destinations:            
            try:
                d.Initialise()
            except:
                traceback.print_exc()
                pass

        #See if we need to launch the web server
        webportno = 0
        enablewebserver = False
        indx = -1
        for d in self.destinations:
            indx += 1            
            try:                
                if d.webendpoint is not None:                    
                    enablewebserver = True
                    self.webcontent[d.webendpoint] = d
                    webport = d.webport
            except:
                traceback.print_exc()
                

        if enablewebserver == True:
            self.logger.info("Initialising MOS Webserver")
            self.webhost = OutputEngineWS(webport,self)
            self.webhost.Start()
            pass

        self.initialised = True
        self.initialising = False

        if len(self.buffer) > 0:            
            for b in self.buffer:            
                self.Set(b['name'],b['value'],options=b['options'],stripparent=b['stripparent'],force=b['force'],config=b['config'])

        self.buffer = None

    def GetName(self,d):
        if 'name' in d.config:
            return d.config['name']

        try:
            return d.output.GetName()
        except:
            st = d.module
            for k in d.config:
                if k == "module":
                    continue
                if k == "name":
                    continue
                if k == "type":
                    continue
                if k == "method":
                    continue

                if st != "":
                    st += ":"
                st += str(d.config[k])

        return st
   
    #Set a value
    def Set(self,name,value,options=None,stripparent=False,force=False,config=None):

        if self.initialising == True:
            self.buffer.append({"name": name, "value": value, "options": options, "stripparent": stripparent,"force": force,"config": config})
            return

        if self.initialised == False:
            self.Initialise()

        sendcount = 0
        failcount = 0

        self.logger.debug("Writing " + name + " = " + str(value) + " into " + str(len(self.destinations)) + " destination")
            
        if force == False:
            try:
                if self.values[name] == value:
                    return
            except:
                pass

        self.values[name] = value
        if isinstance(value,str):
            if options is None:
                options = {"datatype": "text"}
            else:
                if "text" not in options:
                    options['datatype'] = "text"

        if options is None:
            options = {}

        tname = name.replace("\t","")        
        
        for d in self.destinations:            
            try:                
                self.logger.debug("Checking For Validity Of " + str(tname) + " on " + d.cname)
                if d.hierarchy == False:
                    if stripparent == True:
                        bits = name.split("/")
                        tname = "/".join(bits[1:])                        
                    tname = tname.replace("/"," ") 

                if not checkFilters(d,name,options):
                    self.logger.debug("   Not Transmitting - Filtered Out")
                    continue                         
                
                try:
                    self.logger.debug("  Writing Output To " + d.cname)
                    ret = False
                    try:
                        ret = d.Set(tname,value,options)                    
                    except:
                        self.logger.debug("  Transmission Failed. Reinitialising. " + d.cname)
                        try:
                            d.Close()
                        except:
                            pass
                        d.Initialise()
                        ret = d.Set(tname,value,options)

                    if ret is None or ret == True:
                        sendcount += 1
                    else:
                        failcount += 1
                except Exception as e:
                    failcount += 1
                    self.logger.warn("ERROR: Unable to write data: " + str(e))
                    pass

            except:
                traceback.print_exc()
                pass
        
        if config is not None:
            cfg = config
            if isinstance(config,str):
                cfg = json.loads(config)
            if isinstance(cfg,dict):
                cfg = [cfg]
                

            if len(config) > 0:
                #Process additional per-alert configs...
                for x in cfg:                    
                    if 'type' in x:
                        if x['type'].lower() != 'value':
                            continue

                    d = self.LoadDestination(x)
                    #print(str(d))
                    dname = self.GetName(d)

                    if d.hierarchy == False:
                        if stripparent == True:
                            bits = name.split("/")
                            tname = "/".join(bits[1:])                        
                        tname = tname.replace("/"," ") 

                    self.logger.debug("Searching For Existing Module: " + str(dname))

                    #Does this destination already exist?
                    found = False
                    for dx in self.destinations:
                    
                        if dx.cname == dname:
                            self.logger.debug("Using existing connection")
                            try:              
                                ret = False
                                if not checkFilters(d,name,options):
                                    continue
                                try:
                                    ret = d.Set(tname,value,options)                                       
                                except:                                    
                                    self.logger.debug("  Transmission Failed. Reinitialising. " + d.cname)
                                    try:
                                        d.Close()
                                    except:
                                        pass
                                    d.Initialise()
                                    ret = d.Set(tname,value.options)     
                                    
                                if ret is None or ret == True:
                                    sendcount += 1
                                else:
                                    failcount += 1

                            except Exception as e:
                                failcount += 1
                                self.logger.warn("ERROR: Unable to write value: " + str(e))

                            found = True
                            break
                    

                    if found == False:                                                
                        try:
                            self.logger.debug("Using fresh connection")
                            if not checkFilters(d,name,options):
                                continue
                            d.Initialise()
                            ret = d.Set(tname,value,options)
                            if ret is None or ret == True:
                                sendcount += 1
                            else:
                                failcount += 1                            
                            d.Close()
                        except Exception as e:
                            failcount += 1
                            self.logger.warn("ERROR: Unable to write value: " + str(e))
                            pass

        return (sendcount,failcount)
    
    def Stop(self):
        for d in self.destinations:
            try:
                d.Close()
            except:
                traceback.print_exc()
                pass

        if self.webhost is not None:
            self.webhost.Stop()

#Defines an object output path
class OBOutput:
    def __init__(self,name,configfile="output.json",modules="",ignore=[]):
        self.name = name
        self.configfile = configfile
        self.destinations = []
        self.webcontent = {}
        self.batching = None
        self.webhost = None
        self.modules = modules
        if modules == "":
            for q in sys.path:
                final = q + "/ardi/util/outputengine"                
                if os.path.exists(final):
                    modules = final
                    self.modules = final
                    break                                
        else:
            self.modules = modules
        self.values = {}
        self.ignore = ignore
        self.initialised = False
        if self.modules != "":
            self.modules += "/"
        self.LoadAll(configfile)

    def LoadAll(self,configfile):
        filelist = []
        if os.path.isfile(configfile):
            filelist.append(configfile)

        fname = os.path.basename(configfile)
        lastpath = os.path.dirname(configfile)
        pth = os.path.dirname(os.path.dirname(configfile))
        while pth != lastpath:            
            p = pth + "/" + fname
            if os.path.isfile(p):
                filelist.append(p)
            lastpath = pth
            pth = os.path.dirname(pth)

        #print("Using Config Files: " + str(filelist))
        for f in filelist:
            self.Load(f)

    #Load the output configuration
    def Load(self,configfile):
        try:
            fl = open(configfile,'r')
            content = fl.read()
            jsn = json.loads(content)
            for x in jsn:
                if x['method'] in self.ignore:
                    print("This application is not compatible with " + x['method'] + " output.")
                    continue
                if 'type' in x:
                    if x['type'].lower() != 'object':
                        continue
                self.AddDestination(x)
            fl.close()
        except:

            traceback.print_exc()
            print("No Output Configuration File Found")
            pass

    #Load a destination module
    def LoadDestination(self,d):
        method = d['method']        
        library = 'ob_' + method
        try:
            spec = importlib.util.spec_from_file_location(library,self.modules + library+".py")
            module = importlib.util.module_from_spec(spec)
            spec.loader.exec_module(module)
        except:
            traceback.print_exc()                     

        try:
            return module.output(d)
        except:
            traceback.print_exc()
            print(str(dir(module)))
            return None

    #Add a destination
    def AddDestination(self,d):
        try:
            ob = self.LoadDestination(d)
            if ob is not None:

                ob.webendpoint = None
                try:            
                    ob.hierarchy = ob.Supports('hierarchy')
                except:
                    ob.hierarchy = False
                    pass
                
                try:            
                    ob.webendpoint = ob.Supports('html')
                    if ob.webendpoint == False:
                        ob.webendpoint = None

                    if ob.webendpoint is not None:                    
                        if 'port' in d:
                            ob.webport = d['port']
                        else:
                            ob.webport = 8199
                except:
                    ob.webendpoint = None
                    pass

                try:
                    ob.filter = d['filter']
                except:
                    ob.filter = None

                try:
                    ob.allow = d['allow']
                    if isinstance(d['allow'],str):
                        tags = d['allow'].split(' ')
                        ob.allow = tags
                except:
                    ob.allow = None

                try:
                    ob.deny = d['deny']
                    if isinstance(d['deny'],str):
                        tags = d['deny'].split(' ')
                        ob.deny = tags
                except:
                    ob.deny = None

                ob.name = self.name + " " + d['method'].upper() + " Output"

                self.destinations.append(ob)
        except:
            print("Failed to load destination: " + d['method'])
            traceback.print_exc()
            return False
        return True

    #Initialise the system (connect to any data stores)
    def Initialise(self):
        if self.initialised == True:
            return

        for d in self.destinations:
            try:
                d.Initialise()
            except:
                traceback.print_exc()
                pass

        #See if we need to launch the web server
        webportno = 0
        enablewebserver = False
        indx = -1
        for d in self.destinations:
            indx += 1            
            try:                
                if d.webendpoint is not None:                    
                    enablewebserver = True
                    self.webcontent[d.webendpoint] = d
                    webport = d.webport
            except:
                traceback.print_exc()
                

        if enablewebserver == True:
            print("Starting Web Services")            
            self.webhost = OutputEngineWS(webport,self)
            self.webhost.Start()
            pass

        self.initialised = True
   
    #Write an Object Value
    def Write(self,name,value,options=None,stripparent=False,force=False):
        if self.initialised == False:
            self.Initialise()     
            
        if force == False:
            try:
                if self.values[name] == value:
                    return
            except:
                pass

        self.values[name] = value

        if options is None:
            options = []

        for d in self.destinations:            
            try:
                proceed = True
                if d.filter is not None:                    
                    if re.search(d.filter,name) is None:                                                
                        continue

                if d.allow is not None:                  
                    for q in d.allow:                        
                        if q not in options:     
                            proceed = False
                            break

                if d.deny is not None:
                    for q in d.deny:
                        if q in options:
                            proceed = False
                            break

                if proceed == False:
                    continue

                tname = name.replace("\t","")
                if d.hierarchy == False:
                    if stripparent == True:
                        bits = name.split("/")
                        tname = "/".join(bits[1:])                        
                    tname = tname.replace("/"," ") 
                                    
                ret = d.Write(tname,value,options)                
                if ret is None or ret == True:
                    sendcount += 1
                else:
                    failcount += 1
            except:
                failcount += 1
                pass
    
    def Stop(self):
        for d in self.destinations:
            try:
                d.Close()
            except:
                traceback.print_exc()
                pass

        if self.webhost is not None:
            self.webhost.Stop()

#Defines an event log
class EVOutput:
    def __init__(self,name,configfile="output.json",modules="",ignore=[]):
        self.name = name
        self.configfile = configfile
        self.destinations = []
        self.webcontent = {}
        self.batching = None
        self.webhost = None
        self.modules = modules
        if modules == "":
            for q in sys.path:
                final = q + "/ardi/util/outputengine"                
                if os.path.exists(final):
                    modules = final
                    self.modules = final
                    break                                
        else:
            self.modules = modules
        self.values = {}
        self.ignore = ignore
        self.initialised = False
        self.lasttx = None
        self.cooldown = None
        self.mods = {}
        if self.modules != "":
            self.modules += "/"

        self.logger = logging.getLogger(__name__)
        #self.logger.setLevel(logging.DEBUG)
        self.LoadAll(configfile)        

    def LoadAll(self,configfile):
        self.logger.info("Loading MOS Configurations From " + str(configfile))
        filelist = []
        if os.path.isfile(configfile):
            filelist.append(configfile)

        fname = os.path.basename(configfile)
        lastpath = os.path.dirname(configfile)
        pth = os.path.dirname(os.path.dirname(configfile))
        while pth != lastpath:            
            p = pth + "/" + fname
            if os.path.isfile(p):
                filelist.append(p)
            lastpath = pth
            pth = os.path.dirname(pth)

        #print("Using Config Files: " + str(filelist))
        for f in filelist:
            self.Load(f)

    #Load the output configuration
    def Load(self,configfile):
        self.logger.info("Loading Individual MOS Configurations From " + str(configfile))
        try:
            fl = open(configfile,'r')
            content = fl.read()
            jsn = json.loads(content)
            for x in jsn:
                if x['method'] in self.ignore:
                    print("This application is not compatible with " + x['method'] + " output.")
                    continue
                if 'type' in x:
                    if x['type'].lower() != 'event':
                        continue

                self.AddDestination(x)
            fl.close()
        except:
            #traceback.print_exc()
            print("No Output Configuration File Found")
            pass

    #Load a destination module
    def LoadDestination(self,d):
        method = d['method']

        #Try and calculate the final name for this...
        ob = None
        nm = ""
        if 'name' in d:
            nm = d['name']
        else:
            nm = method
            for k in d:
                if k == 'method':
                    continue
                if nm != "":
                    nm += ":"
                nm += str(d[k])

        if method not in self.mods:
            library = 'ev_' + method
            try:
                spec = importlib.util.spec_from_file_location(library,self.modules + library+".py")
                module = importlib.util.module_from_spec(spec)
                spec.loader.exec_module(module)

                self.mods[method] = module
            except Exception as e:
                self.logger.warn("Can't Load " + method + " module - " + str(e))
                traceback.print_exc()                     

            try:
                ob = module.output(d)
                ob.module = method
                ob.config = d
                ob.cname = self.GetName(ob)        
                ob.host = self
            except:
                traceback.print_exc()
                #print(str(dir(module)))
                return None
        else:
            ob = self.mods[method].output(d)
            ob.module = method
            ob.config = d
            ob.cname = self.GetName(ob)
            ob.host = self

        if ob is not None:
            ob.webendpoint = None
            try:            
                ob.hierarchy = ob.Supports('hierarchy')
            except:
                ob.hierarchy = False
                pass
                
            try:            
                ob.webendpoint = ob.Supports('html')
                if ob.webendpoint == False:
                    ob.webendpoint = None

                if ob.webendpoint is not None:                    
                    if 'port' in d:
                        ob.webport = d['port']
                    else:
                        ob.webport = 8199
            except:
                ob.webendpoint = None
                pass

            try:               
                ob.filter = d['filter']
                print("Found Filter: " + ob.filter)
            except:                
                ob.filter = None

            try:
                ob.allow = d['allow']
                if isinstance(d['allow'],str):
                    tags = d['allow'].split(' ')
                    ob.allow = tags
            except:
                ob.allow = None

            try:
                ob.deny = d['deny']
                if isinstance(d['deny'],str):
                    tags = d['deny'].split(' ')
                    ob.deny = tags
            except:
                ob.deny = None

            try:
                ob.cooldown = d['cooldown']                
            except:
                ob.cooldown = None

            ob.name = self.name + " " + d['method'].upper() + " Output"

        return ob

    #Add a destination
    def AddDestination(self,d):
        try:
            ob = self.LoadDestination(d)
            if ob is not None:
                
                self.destinations.append(ob)
                #print("Loaded Destination: " + d['method'])
        except:
            print("Failed to load destination: " + d['method'])
            traceback.print_exc()
            return False
        return True

    #Format a text message
    def Format(self,ob,fmt,message,duration,offset,options = None,sanitise=None):    
        ln = fmt
        en = datetime.datetime.now() - datetime.timedelta(seconds=offset)
        st = en - datetime.timedelta(seconds=duration)
        
        tz = pytz.utc
        try:
            tz = ob.tz
        except:
            pass

        dateformat = "%Y-%m-%d %H:%M:%S"
        try:
            dateformat = ob.dateformat
        except:
            pass

        st = st.astimezone(tz)
        en = en.astimezone(tz)

        st = st.strftime(dateformat)
        en = en.strftime(dateformat)        

        tags = []        
        if options is not None:
            for n in options:
                try:
                    if options[n] == True:
                        tags.append(n)
                                        
                    #self.logger.debug("Replacing " + n.upper() + " with " + str(options[n]))
                    if sanitise is None:
                        ln = ln.replace("%" + n.upper() + "%",str(options[n]))
                    else:
                        ln = ln.replace("%" + n.upper() + "%",str(sanitise(options[n])))
                except:
                    traceback.print_exc()
                    pass

        if len(tags) > 0:
            tgs = ", ".join(tags)
            ln = ln.replace("%TAGS%",tgs)

        du = str(duration)
        if sanitise is None:
            ln = ln.replace("%S",st)
        else:
            ln = ln.replace("%S",sanitise(st))
                            
        if sanitise is None:
            ln = ln.replace("%E",st)
        else:
            ln = ln.replace("%E",sanitise(st))

        ln = ln.replace("%D",du)
        if sanitise is None:
            ln = ln.replace("%M",message)     
        else:
            ln = ln.replace("%M",sanitise(message))     
        
        self.logger.debug("Formatting Message " + fmt + " as " + ln)

        return ln

    #Initialise the system (connect to any data stores)
    def Initialise(self):
        if self.initialised == True:
            return

        for d in self.destinations:
            try:
                d.Initialise()
                d.host = self
            except:
                traceback.print_exc()
                pass

        #See if we need to launch the web server
        webportno = 0
        enablewebserver = False
        indx = -1
        for d in self.destinations:
            indx += 1            
            try:                
                if d.webendpoint is not None:                    
                    enablewebserver = True
                    self.webcontent[d.webendpoint] = d
                    webport = d.webport
            except:
                traceback.print_exc()
                

        if enablewebserver == True:
            print("Starting Web Services")            
            self.webhost = OutputEngineWS(webport,self)
            self.webhost.Start()
            pass

        self.initialised = True    
        
    def GetName(self,d):
        if 'name' in d.config:
            return d.config['name']

        try:
            return d.output.GetName()
        except:
            st = d.module
            for k in d.config:
                if k == "module":
                    continue
                if k == "name":
                    continue
                if k == "type":
                    continue
                if k == "method":
                    continue
                if k == "format":
                    continue
                if k == "dateformat":
                    continue
                if k == "timezone":
                    continue
                if k == "format":
                    continue

                if st != "":
                    st += ":"
                st += str(d.config[k])

        return st
   
    #Write a log entry
    def Write(self,name,duration,offset=0,options=None,stripparent=False,config=None):        
        if self.initialised == False:
            self.Initialise()

        if options is None:
            options = {}        

        try:
            if self.cooldown is not None:
                if self.lasttx is not None:

                    tm = (datetime.datetime.now() - self.lasttx).total_seconds()
                    if tm < self.cooldown:
                        self.logger.info("Event " + name + " skipped due to cooldown.")    
                        return
        except:
            traceback.print_exc()

        sendcount = 0
        failcount = 0

        #print("Writing " + name)

        self.logger.info("Writing Event " + name + " for " + str(duration) + "s into " + str(len(self.destinations)) + " destination")
        self.logger.debug(str(config))

        self.lasttx = datetime.datetime.now()

        #print(str(options))

        for d in self.destinations:            
            try:
                #proceed = True
                if not checkFilters(d,name,options):
                    self.logger.debug("   Not Transmitting - Filtered Out")
                    continue

                tname = name.replace("\t","")
                if d.hierarchy == False:
                    if stripparent == True:
                        bits = name.split("/")
                        tname = "/".join(bits[1:])                        
                    tname = tname.replace("/"," ")                
                                                 
                try:
                    ret = d.Write(tname,duration,offset,options)
                    if ret is None or ret == True:
                        sendcount += 1
                    else:
                        failcount += 1
                except Exception as e:
                    self.logger.warn("Failed to record event: " + str(e))
                    failcount += 1
            except:
                traceback.print_exc()
                pass
        
        #print("Checking Out Per-Alert Configs: " + str(config))
        if config is not None:
            cfg = config
            if isinstance(config,str):
                cfg = json.loads(config)
            if isinstance(cfg,dict):
                cfg = [cfg]

            if len(config) > 0:
                #Process additional per-alert configs...
                for x in cfg:            
                    
                    if 'type' in x:
                        if x['type'].lower() != 'event':
                            continue                    

                    d = self.LoadDestination(x)   
                    
                    #print("Destination: " + str(d))
                    dname = self.GetName(d)

                    tname = name
                    try:
                        if d.hierarchy == False:
                            if stripparent == True:
                                bits = name.split("/")
                                tname = "/".join(bits[1:])                        
                            tname = tname.replace("/"," ") 
                    except:
                        pass

                    #print("Searching For Module: " + str(dname))
                    self.logger.debug("Searching For Existing Module: " + str(dname))                    

                    #Does this destination already exist?
                    found = False
                    for dx in self.destinations:
                    
                        if dx.cname == dname:
                            self.logger.debug("Using existing connection")
                            if not checkFilters(d,name,options):
                                continue
                            try:
                                ret = d.Write(tname,duration,offset,options)
                                if ret is None or ret == True:
                                    sendcount += 1
                                else:
                                    failcount += 1
                            except Exception as e:
                                failcount += 1
                                self.logger.warn("ERROR: Unable to write value: " + str(e))

                            found = True
                            break
                    

                    if found == False:                                                
                        try:
                            self.logger.debug("Using fresh connection")
                            if not checkFilters(d,name,options):
                                continue
                            d.Initialise()                            
                            ret = d.Write(tname,duration,offset,options)
                            if ret is None or ret == True:
                                sendcount += 1
                            else:
                                failcount += 1
                            try:
                                d.Close()
                            except:
                                pass
                            
                        except Exception as e:
                            failcount += 1
                            self.logger.warn("ERROR: Unable to write value: " + str(e))
                            pass
        return (sendcount,failcount)
    
    def Stop(self):
        for d in self.destinations:
            try:
                d.Close()
            except:
                traceback.print_exc()
                pass

        if self.webhost is not None:
            self.webhost.Stop()


class InfoServer(resource.Resource):
    isLeaf = True

    def Message(self,request,message,state,code):
        content = "{"
        content += '  "message": "' + str(message) + '", "state": "' + str(state) + '", "code": ' + str(code)
        content += "}"
        
        request.responseHeaders.addRawHeader('Content-Type', 'application/json')
        request.responseHeaders.addRawHeader('Access-Control-Allow-Origin','*')
        return content.encode('utf-8')    

    #Handle HTTP POST Request
    def render_POST(self, request):
        body = request.content.read()
        return self.render_GET(request,body)

    #Handle HTTP GET Request
    def render_GET(self, request,body = None):
        #print(str(request.uri))
        if request.uri == "/favicon.ico":
            request.setResponseCode(404)            
            return "Not Found.".encode('utf-8')
        
        model = self.model

        uri = request.uri
        if uri[0] == '/':           
            uri = uri[1:]
        
        bits = uri.decode().split('/')
        if bits[0] == '':
            bits = bits[1:]  

        if bits[0] in model.webcontent:
            request.responseHeaders.addRawHeader('Access-Control-Allow-Origin','*')
            try:
                content = model.webcontent[bits[0]].HTML(uri,request)                        
            except:
                traceback.print_exc()
                pass
            return content.encode('utf-8')        
        
        request.setResponseCode(404)            
        return "Not Found.".encode('utf-8')

class OutputEngineWS:
    def __init__(self,portno,oe):
        self.mainthread = None
        self.server = None
        self.portno = portno
        self.oe = oe

    def ThreadFunc(self):
        print("Starting Output Webservices On Port " + str(self.portno))
        info = InfoServer()
        info.model = self.oe
        self.server = server.Site(info)
        reactor.listenTCP(int(self.portno), self.server)
        reactor.run(installSignalHandlers=False)

    def Start(self):
        self.mainthread = threading.Thread(target=self.ThreadFunc,daemon=True)
        self.mainthread.start()

    def Stop(self):
        if self.mainthread is not None:
            reactor.stop()
            pass
        self.mainthread = None

class OutputEngine:
    def __init__(self,name,configfile="output.json",modules="",exclude=[]):
        self.events = EVOutput(name,configfile,modules=modules,ignore=exclude)
        self.keyvalue = KVOutput(name,configfile,modules=modules,ignore=exclude)
        self.objects = OBOutput(name,configfile,modules=modules,ignore=exclude)

    def Start(self):
        self.events.Initialise()
        self.keyvalue.Initialise()
        self.objects.Initialise()

    def Stop(self):
        self.events.Stop()
        self.keyvalue.Stop()
        self.objects.Stop()

    def WriteValue(self,name,value,options=None,stripparent=False,force=False,config=None):                
        try:
            return self.keyvalue.Set(name,value,options,stripparent,force,config=config)
        except:
            traceback.print_exc()

    def WriteLog(self,name,duration=0,offset=0,options=None,stripparent=False,config=None):
        return self.events.Write(name,duration,offset,options,stripparent,config)

    def WriteObject(self,name,value,options=None,config=None):
        self.objects.Write(name,value,options,config)

    def MergeConfig(self,a,b):
        if a is None:
            return b
        if b is None:
            return a

        if isinstance(a,str):
            if isinstance(b,list) or isinstance(b,dict):
                b = json.dumps(b)

            if a[0] == '[':
                if b[0] == '[':
                    return a[0:len(a)-1] + "," +  b[1:len(b)-2]
                else:
                    return a[0:len(a)-1] + "," + b + "]"
            else:
                if b[0] == '[':
                    return b[0:len(a)-1] + "," +  a[1:len(b)-2]
                else:
                    return "[" + a + "," + b + "]"

        if isinstance(a,dict):
            a = [a]

        if isinstance(a,list):

            if isinstance(b,str):
                b = json.loads(b)

            if isinstance(b,dict):
                b = [b]

            for x in b:
                a.append(x)
            
        return a
