import datetime
import traceback
import json 

available = False

try:
    from opcua import ua,uamethod,Server
    available = True
except:
    pass

class output:
    def __init__(self,config):
        self.config = config    
        self.uaindex = 10
        self.opcvars = {}
        self.folders = {}
        self.idpersist = IDPersistance("uaids.json")
        self.port = config['port']
        self.namespace = "http://examples.freeopcua.github.io"
        self.ip = "0.0.0.0"
        if 'namespace' in config:
            self.namespace = config['namespace']
        if 'ip' in config:
            self.ip = config['ip']

    def Initialise(self):        
        try:            
            self.uaserver = Server()

            #Setup UA Server
            self.uaserver.set_endpoint("opc.tcp://" + self.ip + ":" + str(self.port) + "/freeopcua/server")
            self.uaserver.set_server_name(self.name)
            self.uaserver.set_security_policy([
                    ua.SecurityPolicyType.NoSecurity,
                    ua.SecurityPolicyType.Basic256Sha256_SignAndEncrypt,
                    ua.SecurityPolicyType.Basic256Sha256_Sign])

            # setup our own namespace
            uri = self.namespace
            self.idx = self.uaserver.register_namespace(uri)

            print("Starting OPC Server @ Port " + (str(self.port)) + " on IP " + (self.ip))

            self.uaserver.start()
            self.idpersist.Save()
            
            self.opcvars = {}
        except:
            traceback.print_exc()
            return False
        return True

    def GetUANode(self,namespace,name,tryagain=False):
        if tryagain == False:
            return ua.NodeId(identifier=self.idpersist.PointID(name),namespaceidx=namespace)
        else:
            return ua.NodeId(identifier=self.idpersist.TryAgain(name),namespaceidx=namespace)

    def Supports(self,nm):
        #print("Checking for " + nm + " support...")
        if nm == 'hierarchy':
            return True
        return False

    def Set(self,name,value,options=None):
        base = self.uaserver.nodes.objects
        name = name.replace(":","_")
        try:
            if name not in self.opcvars:
                bits = name.split('/')

                fullname = ""

                for q in range(0,len(bits)-1):                    
                    for n in range(0,q+1):
                        if fullname != "":
                            fullname += "/"
                        fullname += bits[n]               
                    if fullname not in self.folders:
                        if q == len(bits)-2:                            
                            base = base.add_object(self.GetUANode(self.idx, fullname),bits[q])
                        else:                            
                            base = base.add_folder(self.GetUANode(self.idx, fullname),bits[q])
                        self.folders[fullname] = base
                    else:
                        base = self.folders[fullname]

                self.idx = self.uaindex
                try:
                    newvar = base.add_variable(self.GetUANode(self.idx, fullname + "/" + bits[len(bits)-1]),bits[len(bits)-1],float(value))
                except:
                    newvar = base.add_variable(self.GetUANode(self.idx, fullname + "/" + bits[len(bits)-1]),bits[len(bits)-1],value)
                self.opcvars[name] = newvar
                self.uaindex += 1

                self.idpersist.Save()
            else:
                self.opcvars[name].set_value(value)
        except:
            traceback.print_exc()

    def Close(self):
        self.uaserver.stop()
        pass


class IDPersistance:
    def __init__(self,filename):
        try:
            fl = open(filename,'r')
            content = fl.read()
            fl.close()

            self.points = json.loads(content)
        except:
            self.points = {}

        self.freepoints = []
        self.filename = filename        
        self.used = []
        self.startfrom = 20
        self.nextfree = 2
        self.avoid = []
        self.changed = False

        self.FindFreePoints()

    def FindFreePoints(self):
        usedids = []
        for n in self.points:
            usedids.append(self.points[n])

        if len(usedids) > 1:
            mx = max(usedids)
        else:
            if len(usedids) == 0:
                mx = 0
            else:
                mx = usedids[0]
        for q in range(self.startfrom,mx):
            if q not in usedids:
                self.freepoints.append(q)

        self.nextfree = mx+1
        if self.nextfree < self.startfrom:
            self.nextfree = self.startfrom

    def GetNextFree(self):
        if len(self.freepoints) > 0:
            val = self.freepoints[0]
            del self.freepoints[0]
            return val

        val = self.nextfree
        self.nextfree += 1
        
        while val in self.avoid:
            val = self.nextfree
            self.nextfree += 1
            
        return val

    def Save(self):
        if self.changed == True:
            content = json.dumps(self.points)
            fl = open(self.filename,'w')
            fl.write(content)
            fl.flush()
            fl.close()
        self.changed = False
                    
    def SaveClean(self):
        scrap = []
        for n in self.points:
            if n not in self.used:
                scrap.append(n)

        for n in scrap:
            del self.points[n]                

        content = json.dumps(self.points)
        fl = open(self.filename,'w')
        fl.write(content)
        fl.flush()
        fl.close()

    def PointID(self,name):
        if name in self.points:
            self.used.append(name)
            return self.points[name]

        self.points[name] = self.GetNextFree()
        self.used.append(name)
        #print("Returning " + str(self.points[name]) + " for " + name)
        self.changed = True
        return self.points[name]

    def TryAgain(self,name):        
        self.points[name] = self.GetNextFree()
        return self.points[name]
