#!/usr/bin/ardipython
import ardiapi as ardi
import json
import requests
import re
import traceback
import time
import threading
import glob
import os
import sys
import importlib
import datetime
import argparse
import platform
import extalerts
import ardi.util.outputengine as outputengine

from pathlib import Path

def IsZeroOrNone(vl):
    try:
        if vl is None:
            return True
        if int(vl) == 0:
            return True
    except:
        pass
    return False

class Calculation:
    def __init__(self):
        self.name = ""
        self.inputaddresses = {}
        self.inputvalues = []
        self.expression = "'^'"
        self.value = None
        self.updated = False

class Alarm:
    def __init__(self):
        self.id = 0
        self.asset = 0
        self.name = ""
        self.expression = None
        self.status = None
        self.flags = 0
        self.inputvalues = []
        self.inputaddresses = {}

    def LoadFromList(self,lst,mgr):
        self.id = lst['id']
        self.name = lst['name']
        self.asset = lst['assetid']
        self.expression = lst['function']
        if int(lst['state']) == 0:
            self.status = False
        else:
            if int(lst['state']) == 1:
                self.status = True
            else:
                self.status = None

        self.expression = self.expression.replace(" = ","==")

        properties = []
        exp = r"\[(.*?)\]"        
        matchset = re.finditer(exp,self.expression, re.MULTILINE)
        x = 0
        for indx,match in enumerate(matchset,start=1):
            for g in match.groups():
                if g == '$$':
                    self.inputvalues.append(None)
                    self.expression = self.expression.replace("[" + str(g) + "]","self.inputvalues[" + str(x) + "]")
                    self.inputaddresses[str(self.asset) + ":-2:" + str(self.id)] = x
                    x = x + 1        

                else:
                    if g == '##':
                        self.inputvalues.append(None)
                        self.expression = self.expression.replace("[" + str(g) + "]","self.inputvalues[" + str(x) + "]")
                        self.inputaddresses[str(self.asset) + ":-3:" + str(self.id)] = x
                        x = x + 1     
                    else:
                        bits = g.split('.')
                        ast = self.asset
                        if len(bits) > 1:
                            ast = int(bits)
                            g = bits[1]
                            
                        properties.append(str(g))
                        self.expression = self.expression.replace("[" + str(g) + "]","self.inputvalues[" + str(x) + "]")
                        self.inputvalues.append(None)
                        node = "text"

                        propid = int(g)
                        #print(str(mgr.props))
                        for prp in mgr.props:
                            #print("Checking " + str(propid) + " vs " + str(prp['id']))
                            if int(prp['id']) == propid:
                                #print('Found Type: ' + prp['type'])
                                if prp['type'] == "MEASUREMENT":
                                    node = "measurement"
                                if prp['type'] == "STATUS":
                                    node = "state"
                                if prp['type'] == "TEXT":
                                    node = "text"
                                if prp['type'] == "ENUM" or prp['type'] == "LOOKUP":
                                    node = "value"
                                break
                        
                        self.inputaddresses[str(ast) + ":" + str(g) + ":" + node] = x
                        x = x + 1        

    def UpdateInput(self,name,value):       
        try:
            value = float(value)
        except:
            if value == "^":
                self.value = "^"
                self.status = None
                self.updated = True
                return self.value
        
        try:           
            self.inputvalues[self.inputaddresses[name]] = value
        except:
            traceback.print_exc()
            pass

        #print("Updating Value: " + str(self.inputvalues))
        v = self.Update()
        try:
            if v != self.value:
                self.value = v
                self.updated = True
        except:
            self.updated = True
            self.value = v

        self.status = v

        if v == '^':
            self.status = None
        else:
            if v != True and v != False:
                if int(v) > 0:
                    self.status = True
                else:
                    self.status = False
            
            if self.flags & 1 > 0:
                if self.status == True:
                    self.status = False
                else:
                    self.status = True
        
        return self.value

    def Update(self):
        try:           
            return eval(self.expression)
        except:
            traceback.print_exc()
            pass
        return "^"

    def __repr__(self):
        state = "Unknown"
        if self.status is not None:
            if self.status == True:
                state = "Active"
            else:
                state = "Inactive"
        return "Alarm: " + self.name + " (" + str(self.id) + ", " + state +")"

class AlarmWriter:
    def __init__(self):
        self.queue = []
        self.access = threading.Lock()
        self.running = True
        self.thread = None
        self.name = ""
        self.server = None

    def AddToQueue(self,itm):        
        self.access.acquire()
        if itm not in self.queue:
            self.queue.append(itm)
        self.access.release()

    def SendItem(self,dta):
        state = "Unknown"
        if dta.status == True:
            state = "ACTIVE"
        if dta.status == False:
            state = "Inactive"
        #print(state + ": " + dta.name)
        pass

    def Start(self):
        if self.thread is None:
            self.thread = threading.Thread(target=self.Loop,daemon=True)
            self.thread.start()

    def Stop(self):
        self.running = False

    def Configuration(self,config):
        pass

    def UpdateStart(self):
        pass

    def UpdateEnd(self):
        pass

    def Loop(self):
        while self.running:
            try:
                self.UpdateStart()
            except:
                pass
            
            while(len(self.queue) > 0):               
                self.access.acquire()
                dta = self.queue[0]
                del self.queue[0]
                self.access.release()
                #print(self.name + " Is Processing Alert " + dta.name)

                try:
                    self.SendItem(dta)
                except:
                    print("Alert Transmission Failed for " + self.name)
                    traceback.print_exc()

            try:
                self.UpdateEnd()
            except:
                pass
            
            time.sleep(1)

class AlarmManager:
    def __init__(self):
        self.server = ""
        self.site = ""
        self.alerts = []
        self.lookup = {}
        self.running = True
        self.writers = []
        self.logfile = ""
        self.sub = None
        self.out = None
        self.outev = None
        pass

    def NewMessage(self,returned,context):
        for ev in returned:
            #print(ev[0])
            if ev[0] == 'alertschanged':
                self.Reload()
                return
        

    def NewData(self,returned,context):
        #print(str(returned))
        for n in returned:
            try:                
                for alrt in self.lookup[n]:
                    st = alrt.status                    
                    alrt.UpdateInput(n,returned[n])                    
                    if alrt.status != st:                        
                        if alrt.status == True:
                            self.TripAlert(alrt)
                        else:
                            if st is not None:
                                self.ResolveAlert(alrt)
                            else:
                                self.BadAlert(alrt)
            except:
                traceback.print_exc()
                pass
            
        #print(str(returned))

    def Reload(self):
        #print("Reloading Due To Change")
        try:
            self.sub.Disconnect()            
        except:
            traceback.print_exc()
        
        self.Connect(self.server,self.site)

    def TripAlert(self,alert):                     
        self.QueueAlert(alert)
        options={"state": 1,"name": alert.name}
        for x in alert.name.split(" "):
            s = x.lower()
            if s != "state":
                options[s] = 1

        try:
            if self.out is not None:
                self.out.Set(alert.name,1,options=options)
        except:
            traceback.print_exc()
        pass

        try:                        
            self.outev.Write(alert.name + " Alert",0,options=options)
        except:
            traceback.print_exc()
        pass

    def BadAlert(self,alert):                     
        self.QueueAlert(alert)
        options={"state": -1,"name": alert.name}
        for x in alert.name.split(" "):
            s = x.lower()
            if s != "state":
                options[s] = 1

        try:
            if self.out is not None:
                self.out.Set(alert.name,1,options=options)
        except:
            traceback.print_exc()
        pass

        try:                        
            self.outev.Write(alert.name + " Offline",1,options=options)
        except:
            traceback.print_exc()
        pass

    def ResolveAlert(self,alert):         
        self.QueueAlert(alert)
        options={"state": 0,"name": alert.name}
        for x in alert.name.split(" "):
            s = x.lower()
            if s != "state":
                options[s] = 1

        try:
            if self.out is not None:
                self.out.Set(alert.name,0,options=options)
        except:
            traceback.print_exc()

        try:
            if self.outev is not None:
                self.outev.Write(alert.name + " OK",0,options=options)
        except:
            traceback.print_exc()
        pass

    def QueueAlert(self,alert):
        #print(str(self.writers))
        for wrtr in self.writers:
            wrtr.AddToQueue(alert)

    def DataWriteThreads(self):
        for wrtr in self.writers:
            wrtr.Start()    

    def Subscribe(self):
        #print("Subscribing!")
        if self.sub is not None:
            self.sub.Disconnect()
            self.sub = None
            
        self.sub = ardi.Subscription(self.srv)
        added = []
        for a in self.alerts:
            for q in a.inputaddresses:
                if q not in added:
                    #print("Subscribing To: " + q)
                    self.sub.AddCode(q)
                    added.append(q)
        
        self.sub.SetCallback(self.NewData,self)
        self.sub.SetMessageCallback(self.NewMessage,self)
        self.sub.Connect()        
        pass

    def Log(self,msg):
        try:
            fl = open(self.logfile,'a')
            tm = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            fl.write(tm + "\t" + msg + "\n")
            fl.flush()
            fl.close()
        except:
            pass

    def LoadWriters(self):        
        if len(self.writers) == 0:            
            writers = []
            pth = os.path.dirname(os.path.abspath(__file__))
            for fl in glob.glob(pth + "/write_*.py"):
                writers.append(Path(fl).name)

            url =  "http://" + self.server + '/s/' + self.site + '/api/alert/settings?format=json'
            resp = requests.get(url)
            writeroptions = {}
            if resp.status_code == 200:            
                writeroptions = resp.json()

            for fl in writers:
                final = fl.replace(".py","")
                self.Log("Loading Writer Module " + fl)
                try:
                    themodule = importlib.import_module(final)
                    newwriter = themodule.Writer()
                    basicname = final[6:]
                    try:
                        newwriter.Configuration(writeroptions[basicname])
                    except:
                        pass

                    newwriter.name = basicname
                    newwriter.server = self
                    self.writers.append(newwriter)                
                except:
                    traceback.print_exc()
                    self.Log("Failed to load Writer Module " + final)
                
            #sys.exit()
            self.DataWriteThreads()
            self.StartExternal()

    def StartExternal(self):
        self.ext = extalerts.ExternalAlertSystem("http://" + self.server + '/s/' + self.site,self)
        self.ext.Start()

    def UpdateAlertList(self):
        url =  "http://" + self.server + '/s/' + self.site + '/api/alert/allalerts?format=json'
        resp = requests.get(url)
        
        self.alerts = []
        
        # HTTP response code, e.g. 200.
        if resp.status_code == 200:            

            config = resp.json()
            for a in config:
                alm = Alarm()
                alm.LoadFromList(a,self)
                self.alerts.append(alm)

        lookup = {}
        for a in self.alerts:
            for v in a.inputaddresses:
                if v not in lookup:
                    lookup[v] = []
                lookup[v].append(a)

        self.lookup = lookup
        print(str(len(self.alerts)) + " Alerts Loaded.")

        self.Subscribe()

    def Connect(self, servername,sitename):
        try:
                logfolder = os.environ['ARDILogPath']
        except:
                if (platform.system() == 'Windows'):
                    logfolder = "c:\\windows\\temp"
                else:
                    logfolder = '/var/log/ardi'

        if logfolder[len(logfolder)-1] != '/':
            logfolder = logfolder + "/"
            
        self.logfile = logfolder + "alarmmanager_" + sitename + ".log"
        print("Alert Manager Started & Logging To " + self.logfile)
        
        self.srv = ardi.Server(servername,sitename)
        while True:
            try:
                self.srv.Connect()
                break
            except:
                time.sleep(5)
                
        self.server = servername
        self.site = sitename

        bits = self.srv.GetConfiguration()
        self.rels = bits[0]
        self.props = bits[1]

        self.LoadWriters()

        configfile = os.path.dirname(__file__) + os.path.sep + self.site + ".json"

        self.out = outputengine.KVOutput(self.site + " Alert Manager",configfile=configfile)
        self.outev = outputengine.EVOutput(self.site + "Alert Manager",configfile=configfile)
        
        self.UpdateAlertList()

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Process ARDI Server Alerts")   
    parser.add_argument('server', help='The URL to the ARDI server')
    args = parser.parse_args()

    mgr = AlarmManager()

    site = "default"
    bits = args.server.split('/')
    if len(bits) > 1:
        site = bits[len(bits)-1]
        
    mgr.Connect(bits[0],site)    
    mgr.out.Stop()
    print("Clean Close")
