import requests
import threading
import time
import traceback

class ExternalAlert:
    def __init__(self,idno,assetno,name,value):
        self.id = idno
        self.name = name
        self.asset = assetno
        self.status = value
        self.flags = 4

class ExternalSource:
    def __init__(self,core,idno,url):
        self.id = idno
        self.url = url
        self.active = {}
        self.lookup = {}
        self.alertobjects = {}
        self.name = ""
        
        self.core = core
        self.newalerts = {}
        self.regex = None
        self.first = True

    def Fetch(self):
        try:
            #print("Scraping: " + self.url)
            resp = requests.get(self.url)
        except:
            print("Failed to retrieve " + str(self.url))
            return False

        alerts = {}
        clear = []
        
        lines = resp.text.replace("\r\n","\n").split("\n")        
        for l in lines:
            bits = l.split("\t")
            if len(bits) <= 1:
                continue

            bits[0] = bits[0].strip()            
               
            if self.IsActive(bits[1]) == True:                    
                try:
                    alerts[bits[0]] += "," + bits[1]
                except:
                    alerts[bits[0]] = bits[1]
            else:
                clear.append(bits[0]);

        for q in alerts:
            self.AddAlert(q,alerts[q])

        for q in clear:
            if q in self.active or self.first == True:
                self.RemoveAlert(q)

        keys = []
        for n in self.active.keys():
            keys.append(n)

        for n in keys:
            if n not in alerts:
                if n not in clear:                   
                    self.RemoveAlert(n)


        self.Process()
        self.first = False
        return True

    def IsActive(self,st):
        st = st.strip()
        if st == "":
            return False
        if st == "0":
            return False
        if st == "None":
            return False
        if st == "null":
            return False

        return True

    def RemoveAlert(self,st):
        try:
            del self.active[st]
        except:
            pass
        
        if st not in self.lookup:
            self.newalerts[st] = ""
            return
        
        self.core.RemoveAlert(self.alertobjects[self.lookup[st]])        

    def AddAlert(self,st,vl):
        self.active[st] = vl
        if st not in self.lookup:
            self.newalerts[st] = vl
            return

        indx = self.lookup[st]
        if indx not in self.alertobjects:
            self.alertobjects[indx] = ExternalAlert(self.id,indx,self.name,vl)
            
        ob = self.alertobjects[indx]
        if ob.status != vl:
            ob.status = vl
            self.core.AddAlert(ob)

    def Process(self):
        qry = {}
        content = []
        for k in self.newalerts:
            content.append(k)

        if len(content) > 0:
            mymap = self.core.MapNamesToID(content,self.id)
            
            for k in mymap:
                indx = int(mymap[k])
                self.lookup[k] = indx
                if indx not in self.alertobjects:
                    self.alertobjects[indx] = ExternalAlert(self.id,indx,self.name,self.newalerts[k])
                self.core.AddAlert(self.alertobjects[indx])

        self.newalerts = {}

class ExternalAlertSystem:
    def __init__(self,url,par):
        self.url = url
        self.sources = []
        self.running = True
        self.thread = None
        self.core = par

    def Start(self):
        self.running = True
        if self.LoadConfig() == False:
            return False

        self.thread = threading.Thread(target=self.Loop,daemon=True)
        self.thread.start()
        
        return True

    def Stop(self):
        running = False

    def Loop(self):        
        while self.running:
            try:
                self.Poll()
            except:
                traceback.print_exc()
                pass
            time.sleep(5)

    def LoadConfig(self):
        url =  self.url + '/api/alert/external?format=json'
        try:
            resp = requests.get(url)
        except:
            print("Server doesn't support external alert services.")
            return False

        content = resp.json()
        for n in content:
            ur = n['url']
            idno = n['idno']
            src = ExternalSource(self,idno,ur)
            src.name = n['name']
            self.sources.append(src)

        return True

    def RemoveAlert(self,alrt):
        alrt.status = ""
        self.core.QueueAlert(alrt)
        #print("Removing " + str(alrt.asset) + ":" + str(alrt.id) + " From Active Alerts")

    def AddAlert(self,alrt):
        #print("Adding " + str(alrt.asset) + ":" + str(alrt.id) + " To Active Alerts With " + str(alrt.status))
        self.core.QueueAlert(alrt)

    def Poll(self):
        for src in self.sources:
            src.Fetch()

    def MapNamesToID(self,lst,sourceid):
        url =  self.url + '/api/alert/lookupnames'
        #print("Looking Up From " + str(url))
        try:
            resp = requests.post(url,{"names": ",".join(lst),"sourceid": sourceid})
        except:            
            return {}

        content = resp.json()

        mymap = {}
        for x in content:
            mymap[x] = content[x]

        #print(str(mymap))
        return mymap
