#!/usr/bin/python

## @package ardi.consolidator.subscription
#
# Handles Requests for Live Data

from twisted.internet import reactor,task
from twisted.web import server,resource
from twisted.web.static import File

import logging
import string
import random
import time
from datetime import datetime

def AsBytes(st):
    try:
        return st.encode('utf-8')
    except:
        return st

def GetArgument(arr, name):
    dta = arr[AsBytes(name)][0]
    try:
        dta = dta.decode('utf-8')
    except:
        pass
    return dta

## Provide a debug message at the server root.
class Root(resource.Resource):
    isLeaf = False
    def render_GET(self,request):
        return AsBytes('ARDI Subscription Server')

## Provide a simple message at the server root.
class BaseRequest(resource.Resource):
    isLeaf = False
    def render_GET(self,request):
        return AsBytes('ARDI Subscription Server<br/><br/>For details on accessing this service, see your ARDI documentation.')

## Corresponds to a single client subscription and long polling request.
class Subscription:
    def __init__(self):
        ## The ARDI links included in the subscription
        self.codes = ()
        ## The unique subscription key ID
        self.id = ""
        ## The time of the most recent record that was returned.
        self.lastresult = 0
        ## Used to time-out the current request if it runs over-long
        self.timeout = 0
        ## The time of the last response.
        self.lastuse = 0
        ## The time the session was initiated
        self.inittime = time.time()

    ## Generates a unique subscription ID.
    def generateid(self):
        self.id = ''.join(random.choice(string.ascii_uppercase + string.digits) for unused in range(8))
        #self.id = "8888"
        return self.id

## Formats a Subscription Response
#
#  This function responds to the client with a JSON or XML message.
#
#   @param sub The Subscription instane to report on
#   @param core The Core containing shared service variables
#   @param format 'json' for JSON formatted data, otherwise XML
#   @param request The request object to allow the information to be returned to the client.
def SubscriptionResponse(sub,core,format,request):
    t = time.time()

    #print("Responding!")
    try:
        format = format.decode('utf-8')
    except:
        pass
    resp = ""
    if format == 'json':
        request.setHeader("content-type","text/json")
        resp += "{\r\n   \"id\": \"" + sub.id + "\",\r\n"
        resp += "   \"items\": [\r\n"
        first = True
        for code in sub.codes:
            v = None
            if code in core.points:
                if (core.pointimes[code] > sub.lastresult) or (sub.lastresult==0):
                    v = core.points[code]
            if v is not None:
                if first == True:
                    first = False
                else:
                    resp += ",\r\n"
                resp += "    { \"code\": \"" + code + "\",\r\n     \"value\" : \"" + v + "\" }"

        resp += "   ],\r\n"
        resp += "   \"messages\": [\r\n"
        first = True
        for msg in core.messagequeue:
            if msg.time > sub.lastresult and msg.time > sub.inittime:
                if first == True:
                    first = False
                else:
                    resp += ",\r\n"
                resp += "   { \"code\": \"" + msg.name[1:] + "\",\r\n     \"value\" : \"" + msg.value + "\" }"
        resp += "   ] }"

    if format == 'xml':
        #print("Update Request...")
        starttime = datetime.now()
        request.setHeader("content-type","text/xml")
        resp += "<update>\r\n  <id>" + sub.id + "</id>\r\n  <points>"
        lines = []
        for code in sub.codes:
            v = None
            if code in core.points:
                if (core.pointimes[code] > sub.lastresult) or (sub.lastresult==0):
                    v = core.points[code]
            if v is not None:
                lines.append("  <point code=\"" + code + "\" value=\"" + v + "\"/>");
                #resp += "  <point code=\"" + code + "\" value=\"" + v + "\"/>"

        respend = "\r\n  </points>\r\n"
        respend += "  <messages>\r\n"

        for msg in core.messagequeue:
            if msg.time > sub.lastresult:
                if msg.time > sub.inittime:
                    respend += "   <message code=\"" + msg.name[1:] + "\" value=\"" + msg.value + "\"/>"
        respend += "\r\n  </messages>\r\n</update>"

        resp += "\r\n".join(lines)+respend
        totaltime = datetime.now() - starttime
        #print("Update Timing: {}".format(totaltime))

    sub.lastresult = t
    #return ToByteString(resp)

    return AsBytes(resp)

## Cross Domain XML
#
#  The cross-domain XML file is used by both Flash and Unity to validate that
#  programs are permitted to connect to these servers.
class CrossDomain(resource.Resource):
    isLeaf = True

    def render(self,request):
        request.setHeader("Content-Type","text/xml")
        return AsBytes("<?xml version=\"1.0\"?>\r\n<cross-domain-policy>\r\n<allow-access-from domain=\"*\"/>\r\n</cross-domain-policy>")

## Processes an Unsubscribe Request
#
#  Allows clients to gracefully unsubscribe from an update feed.
class UnsubscribeRequest(resource.Resource):
    isLeaf = True

    def __init__(self,core):
        self.core = core
        resource.Resource.__init__(self)

    def render(self,request):

        request.setHeader("Access-Control-Allow-Origin","*")
        try:
            subid = GetArgument(request.args,'id')#request.args[AsBytes('id')][0]
        except:
            return AsBytes('Invalid Subscription ID')

        sub = None
        for s in self.core.subscriptions:
            if s.id == subid:
                sub = s
                break

        if sub is None:
            return 'Invalid Subscription ID'

        self.core.subscriptions.remove(sub)
        return AsBytes('OK')

## Subscribes to a set of live data points
#
#  Clients use "subscribe" to create a new subscription for one or more data points.
class SubscribeRequest(resource.Resource):
    isLeaf = True

    def __init__(self, core):
        self.core = core
        resource.Resource.__init__(self)

    def render(self, request):
        request.setHeader("Access-Control-Allow-Origin","*")
        if len(request.args) > 0:
            #try:
                format = "xml"
                try:
                    format = GetArgument(request.args,'format')#request.args[AsBytes('format')][0]
                except:
                    pass

                codes = GetArgument(request.args,'codes')#request.args[AsBytes('codes')][0]
                codes = codes.split(',')
                sub = Subscription()
                sub.codes = codes
                sub.lastuse = time.time()
                sub.lastresult = 0
                sub.timeout = 28

                unique = False
                while unique==False:
                    sub.generateid()
                    unique = True
                    for s in self.core.subscriptions:
                        if s.id == sub.id:
                            unique = False
                            break

                logging.info("New Subscription: " + sub.id)
                self.core.subscriptions.append(sub)
                return SubscriptionResponse(sub,self.core,format,request)
            #except:
            #    pass

        return AsBytes('Error')

## Returns an instantanious list of current values
#
#  This returns a list of data points as they are right now, without initiating a subscription.
class SnapshotRequest(resource.Resource):
    isLeaf = True

    def __init__(self, core):
        print("Starting Snapshot Request")
        self.core = core
        resource.Resource.__init__(self)

    def render(self, request):
        #logging.error("Processing Snapshot...")
        request.setHeader("Access-Control-Allow-Origin","*")
        #print str(request.args)
        if len(request.args) > 0:
		                
            #try:
                format = "xml"
                try:
                    format = GetArgument(request.args,'format')#request.args[AsBytes('format')][0]
                except:
                    pass

                #codeline = request.args[AsBytes('codes')][0]
                #codes = codeline.decode('utf-8').split(',')
                codes = GetArgument(request.args,'codes')
                codes = codes.split(',')
                sub = Subscription()
                sub.codes = codes
                sub.lastuse = time.time()
                sub.lastresult = 0
                sub.timeout = 28
                sub.id = "*"

                #logging.info("Snapshot Request")
                #self.core.subscriptions.append(sub)
                return SubscriptionResponse(sub,self.core,format,request)
            #except:
            #    pass

        return AsBytes('Error (Snapshot) ' + str(request.args))

## Updates an existing subscription
#
#  Clients use "update" to initiate a long-polling request asking for changes to the
#  data they recently subscribed to.
class UpdateRequest(resource.Resource):
    isLeaf = True
    def __init__(self,core):
        self.checkdelay = 1
        self.longpolling = []
        self.core = core

        loopingCall = task.LoopingCall(self.processLongPoll)
        loopingCall.start(self.checkdelay,False)

        resource.Resource.__init__(self)

    ## Checks to see if any data is ready for a given subscription.
    def checkDataReady(self,subid,mark):
        t = time.time()

        sub = None
        for s in self.core.subscriptions:
            #print "  Checking Against " + s.id
            if s.id == subid:
                sub = s
                break

        #Return if the subscription ID is invalid
        if sub is None:
            return None

        if mark == True:
            sub.lastuse = t

        #Timeout to allow many clients to fail gracefully
        if sub.lastuse + sub.timeout < t:
            return (sub,True)

        #print(str(self.core.pointimes))
        #Check to see if any updates have occured
        try:
        	for key, value in self.core.pointimes.items():
        		if value > sub.lastresult and key in sub.codes:                
        			return (sub,True)
        except:
        	for key, value in self.core.pointimes.iteritems():
        		if value > sub.lastresult and key in sub.codes:                
                            return (sub,True)

        #Are there any pending messages for me...
        for msg in self.core.messagequeue:
            if msg.time > sub.lastresult:
                print("New Message Found - Data Is Ready")
                return (sub, True)

        return (sub,False)

    ## Renders the result of the update.
    def render(self,request):

        request.setHeader("Access-Control-Allow-Origin","*")
        try:
            subid = GetArgument(request.args,'id')#request.args[AsBytes('id')][0]
        except:
            return AsBytes('No ID Specified')

        dataready = self.checkDataReady(subid,True)
        if dataready is None:
            return AsBytes('Invalid Subscription ID')

        if self.core.maxusers > 0:
            if len(self.longpolling) >= self.core.maxusers:
                print("Rejected Request - User Count Exceeded" )
                return AsBytes('Too Many Users')
            else:
                print("User Count: " + str(len(self.longpolling)) + " / " + str(self.core.maxusers))

        #print "Got Request..."

        if dataready[1] == True:
            try:
                format = GetArgument(request.args,'format')#request.args['format'][0]
            except:
                format = "xml"

            return SubscriptionResponse(dataready[0],self.core,format,request)
        else:
            #print "Pushing request to long polling..."
            self.longpolling.append(request)
            return server.NOT_DONE_YET

    ## Goes through outstanding long-polling requests.
    #
    # This function forces responses for long-running polls, checks to see if new data is available for
    # all existing, valid polls, and lapses old subscriptions.
    def processLongPoll(self):
        for request in self.longpolling:

            subid = GetArgument(request.args,'id')#request.args[AsBytes('id')][0]

            dataready = self.checkDataReady(subid,False)

            if dataready is None:
                try:
                    request.write("Subscription ID Lapsed")
                    request.finish()
                except:
                    pass

                self.longpolling.remove(request)
                continue

            if dataready[1] == False:
                continue

            #print "Preparing Subscription Response..."
            try:
                try:
                    format = GetArgument(request.args,'format')#request.args['format'][0]
                except:
                    format = "xml"
                request.write(SubscriptionResponse(dataready[0],self.core,format,request))
                request.finish()
            except:
                print("Lost Long Polling Request!")
                pass
            finally:
                self.longpolling.remove(request)

class subscriber:
    def __init__(self):
        self.id = ""
        self.subscribedto = ()

def LogFunction(lf):    
    pass

def GetService(core):
    root = Root()
    root.putChild(AsBytes(""),BaseRequest())
    root.putChild(AsBytes("subscribe"),SubscribeRequest(core))
    root.putChild(AsBytes("update"),UpdateRequest(core))
    root.putChild(AsBytes("unsubscribe"),UnsubscribeRequest(core))
    root.putChild(AsBytes("snapshot"),SnapshotRequest(core))
    root.putChild(AsBytes("crossdomain.xml"),CrossDomain())
    site = server.Site(root)
    
    #This will eat the access log messages that we don't really want to keep.
    site.log = LogFunction
    return site
