## @package ardi.driver
#  Enables rapid creation of ARDI drivers
#
#  These classes are designed to assist in rapidly creating new data drivers.
#
#  DriverBase and DriverCore are used to create LIVE drivers.
#
#  HistDriverBase and HistDriverCore are - obviously - used to create HISTORIAN drivers.

## @package ardi.driver.driverbase
#  ARDI Live Driver Service
#
#  The DriverBase class acts as the 'host' to a number of DriverCores, each of which communicate
#  on the same protocol.
#
#  Each core is allocated its own thread to avoid issues with one device causing problems with others.

import sys
import platform
import logging
import threading
import ardi.driver.driverhttp as driverhttp
import ardi.driver.drivercore as drivercore
import ardi.driver.logsystem as logsystem
import argparse
#import pycurl
import xmltodict
import time
import traceback
import requests
import os
import json


##Thread body for a single Source.
#
#    This is a basic utility function to launch a single 'source' inside this driver.
#
#    \param core: The core that this driver will use
#    \param servername: The name/IP of the system that this driver will report to
#    \param assetid: The ARDI asset ID number of the source
def corethread(core, servername, assetid,context):
    core.Launch(servername, assetid,context)

##Launches a web service for IPC.
#
#   This is a basic utility function to launch a single 'source' inside this driver.
#
#   @deprecated This function is no longer required, as it has been integrated into the DriverBase.
def webservice(core):
    core.logger.info('Starting IPC Service')
    import ardi.driver.ipcservice as ipcservice
    
    ipc = ipcservice.DriverIPC()
    ipc.core = core
    ipc.port = core.hostport
    if ipc.port == 0:
        ipc.port = 8081
        
    try:
        ipc.start()
    except KeyboardInterrupt:
        pass    

## Structure for source details
#
#  This class is used as a structure to hold details about a single ARDI data source.
class ardisourcedetail:
    def __init__(self):
        self.id = 0
        self.subport = 5336
        self.subhost = "localhost"
        self.mode = 0
        self.context = 1

##The base class for a driver.
#
# This class is responsible for all stages of driver life-cycle.
#
# It also communicates directly with ARDI to get details on what data sources it should serve via HTTP web requests
#
# The driver base runs as a single process. It contains multiple threads, each dedicated to a single source.
class ardidriver:

    def __init__(self):
        ##The port on which this drivers IPC interface will run
        self.hostport = 8081        

        ##Used for future 3rd party integration
        self.base = ''

        ##The shell to be used to execute scripts (mostly required to launch Python in Windows)
        self.shell = ''        

        ##The list of currently running core threads
        self.threads = []

        ##The list of local assets
        self.assets = []

        ##Local log lines
        self.basiclog = []
        self.detaillog = []

        self.tracing = None


    ## Prepare Logging
    #        
    # This function sets up logging (with os-specific paths) for the rest of the driver.

    def initlogger(self):
        
        if hasattr(self,'logger'):
            return     
            
        fname = "/var/log/ardi/driver-" + str(self.hostport) + ".log"
        if (platform.system() == 'Windows'):
        	try:
        		logfolder = os.environ['ARDILogPath']
        	except:
        		logfolder = "c:\\windows\\temp"
                
        	fname = logfolder + "\\ardi-driver-" + str(self.hostport) + ".log"
            
        self.logger = logsystem.CreateLogger(self,"Driver " + str(self.hostport),fname)

        self.LogBasic('----DRIVER RESTART----')
        self.LogBasic('Starting Driver')

        redir = logsystem.iologger(self.logger,logging.ERROR)
        sys.stderr = redir

    ## Start IPC service
    #        
    # This function starts the IPC/web service allowing communication between ARDI and the driver.
    #
    # It is primarily used to re-load the driver when a change to the incoming data occurs.

    def startWebService(self):
        self.LogBasic('Starting IPC Service')
        import ardi.driver.ipcservice as ipcservice
        
        self.ipc = ipcservice.DriverIPC()
        self.ipc.core = self
        self.ipc.port = self.hostport
        if self.ipc.port == 0:
            self.ipc.port = 8081
            
        try:
            self.ipc.start()
        except KeyboardInterrupt:
            #print "Done!"
            pass    

    ## Start the Driver
    #
    #   This function is called once all initialisation is complete. It launches sources and the IPC service.
    #
    #   \param driverfactory: The connector factory class
    #   \param port: The port that the server is to run off (deprecated)
    def start(self,driverfactory, port=None):
        
        #self.initlogger()

        ##The driver factory is a class that produces new driver instances        
        self.driverfactory = driverfactory

        self.LoadConfig()
        
        self.startWebService()
        
        self.StopDrivers()

    ## Loads driver configuration from the server
    #   
    #   This function gathers command-line arguments and then loads configuration details from the web.
    def LoadConfig(self):
        
        arglist = sys.argv
        if len(arglist) >= 2:
            if arglist[1] == 'start':                
                arglist = arglist[2:]
            else:
                arglist = arglist[1:]

        parser = argparse.ArgumentParser(description='ARDI Driver')
        parser.add_argument('driverid',metavar='PortNo',type=int, help='Unique ID/port for the driver', nargs='?')
        parser.add_argument('server',metavar='Server', help='Server Name', nargs='?', default='localhost')
        parser.add_argument('assetid',metavar='AssetNo',type=int, help='Asset IDs to act as a driver for', nargs='?')
        parser.add_argument('--multiple',dest='multi',action='store_const',const=True,default=False,help='Allows this driver to connect to multiple devices')        
        parser.add_argument('--script','-r',dest='script',default='')
        parser.add_argument('--shell','-l',dest='shell',default='')
        parser.add_argument('--config','-c',dest='configfile',default='')        
        parser.add_argument('--debuglevel',dest='debuglevel',default=2)
        args = parser.parse_args(arglist)
        
        #self.logger.debug(args)
        
        servername = args.server
        self.hostport = int(args.driverid)
        self.servername = servername
        
        self.initlogger()
        
        self.LogBasic('Driver ' + str(self.hostport) + ' feeds server ' + servername)

        ##The script that should be run on every successful data fetch
        self.script = args.script
        self.shell = args.shell        

        ##A list of all currently running driver instances
        self.drivers = []

        ##A dictionary that allows fast association of asset IDs to data points
        self.assetids = []
        if args.assetid != None:
            for assetid in args.assetid:
                self.assetids.append(assetid)
        else:
            #response = StringIO()

            sleeptimer = 10
            while (True):
                #try:
            
                fullurl = "http://" + servername + "/api/driverdata?port=" + str(self.hostport)

                self.LogDetail("Fetching " + fullurl)

                try:                    
                    web = requests.get(fullurl,timeout=5)
                    break
                except:
                    traceback.print_exc()
                    sleeptimer += 10
                    if (sleeptimer > 60):
                        sleeptimer = 60

                    self.logger.info("ARDI Not Responding. Trying again in " + str(sleeptimer) + " seconds")

                    time.sleep(sleeptimer)                    
                finally:
                    pass				
                    #c.close()

            self.parseconfig(web.text)
            
            self.LogBasic("Load Default Configuration From Server")
        
        #for assetid in self.assetids:
        for asset in self.assets:
            self.LogBasic("Creating driver instance for " + str(asset.id))
            instance = self.driverfactory.createinstance()
            core = drivercore.ardicore(instance)
            core.base = self
            core.source = args.configfile

            subtarget = asset.subhost
            if (asset.subhost == "localhost"):
                #Strip out junk and copy the IP of the original host...
                
                subtarget = servername
                ps = subtarget.find('/');
                if (ps != -1):
                    subtarget = subtarget[0:ps]

                ps = subtarget.find(':');
                if (ps != -1):
                    subtarget = subtarget[0:ps]
                    
            asset.subhost = subtarget
                    
            core.info = asset

            self.logger.info("Source reports to " + str(asset.subhost) + ":" + str(asset.subport) + " & " + str(servername));
            
            self.startcorethread(core, servername, asset.id,asset.context)
            self.drivers.append(core)
            #self.drivers[str(asset.id)] = core

        #print self.drivers
        #webservice(self)

    ## Imports the configuration XML file
    #
    #  This function loads the list of data points within the XML text provided. The function is usually called from LoadConfig
    #
    #   \param xml: The XML text to process
    def parseconfig(self,xml):
        #print xml
        result = xmltodict.parse(xml)

        self.lookups = {}
        try:
            if result['driverdata']['lookups'] == 'present':
                self.loadLookups()
        except:
            pass

        try:
            #Core Parameters
            if '@id' in result['driverdata']['source']:
                self.assetids.append(result['driverdata']['source']['@id'])
                
                ast = ardisourcedetail()                
                ast.id = result['driverdata']['source']['@id']
                ast.subport = result['driverdata']['source']['@subport']
                ast.subhost = result['driverdata']['source']['@host']
                ast.mode = result['driverdata']['source']['@mode']
                ast.context = result['driverdata']['source']['@profile']
                
                self.assets.append(ast)
            else:
                for itm in result['driverdata']['source']:                    
                    self.assetids.append(itm['@id'])
                    
                    ast = ardisourcedetail()                
                    ast.id = itm['@id']
                    ast.subport = itm['@subport']
                    ast.subhost = itm['@host']
                    ast.mode = itm['@mode']
                    ast.context = itm['@profile']   
                    self.assets.append(ast)
        except:
            #print "Exception Processing"
            #traceback.print_exc()
            pass

    ## Load lookup tables
    #
    #  If the ARDI server uses lookup tables, this function downloads copies of all tables.
    #
    #  This allows the driver to perform lookup translation, sparing the web server excessive usage.
    def loadLookups(self):
        self.lookups = {}
        self.LogBasic("Loading Lookup Table")
        try:
            #response = StringIO()
        
            fullurl = "http://" + self.servername + "/api/getlookuptables"
        
            #c = pycurl.Curl()
            #c.setopt(c.URL,fullurl)
            #c.setopt(c.WRITEFUNCTION,response.write)
            #c.setopt(c.TIMEOUT,5)
            #c.setopt(c.CONNECTTIMEOUT,5)
            #c.perform()
            #c.close()
            web = requests.get(fullurl)			
        except:
            self.logger.exception("Failed to load lookup values from server")
            return

        #print response.getvalue()

        currentid = 0
        lines = web.text.splitlines()
        for l in lines:
            bits = l.split("\t")
            if len(bits) == 1:
                bits = l.split(",")
            if len(bits) == 1:
                if l[0] == '!':
                    currentid = int(l[3:])
                    continue
            else:
                if bits[1] == "":
                    continue
                try:
                    self.lookups[currentid][bits[0]] = bits[1]
                except:
                    self.lookups[currentid] = {}
                    self.lookups[currentid][bits[0]] = bits[1]
    ##Stops all of the Cores under this base driver
    #
    #    This function is primarily used during driver shutdown or restart
    def StopDrivers(self):
        
        for drv in self.drivers:
            #self.drivers[drv].Stop()
            drv.Stop();
        
        self.assets = []
        self.drivers = []
        self.threads = []
        self.LogBasic('All Drivers Stopped')
        pass

    ## Starts a single Core thread
    #    
    #    \param core: The core to run in this thread
    #    \param servername: The name of the server to report all data to
    #    \param assetid: The asset ID number of the source
    def startcorethread(self,core, servername, assetid,context):
        t = threading.Thread(target=corethread, args = (core, servername, assetid,context))
        t.daemon = True
        t.start()
        #self.threads[int(assetid)] = t

    ## Pauses a given data source

    #    This function is called from the drivers IPC service

    #    \param assetid: The ID of the source that you wish to reload"""
    def Pause(self, assetid,context):
        self.LogBasic('Pause Requested via IPC')
        for drv in self.drivers:
            if drv.asset == assetid and drv.context == context:
                drv.Pause()
        #self.drivers[assetid].Pause()

    ##Reload a single source
    #
    #   This function is called from the IPC service, usually due to a change in the underlying ARDI
    #    data bindings - such as the addition of a new address, or change of an existing one.
    #
    #    \param assetid: The ID of the source that you wish to reload"""
    def Reload(self, assetid,context):
        for drv in self.drivers:
            if drv.asset == assetid and int(drv.context) == int(context):
                self.LogBasic('Source Reload Requested via IPC On Source ' + str(assetid))
                drv.Reload()
##        if assetid in self.drivers.keys():
##            self.LogBasic('Source Reload Requested via IPC On Source ' + str(assetid))
##            self.drivers[assetid].Reload()
##        else:
##            self.logger.error('Source Reload Requested on Invalid Source ( ' + str(assetid) + " )")

    ## Forces all sources to reload and restart
    #
    #    This function is called from the drivers IPC service"""
    def Restart(self):
        self.LogBasic('Driver Restart Requested via IPC')
        self.StopDrivers()
        self.assets = []
        self.LoadConfig()
        pass

    ## Requests that a new source be started
    #
    #    This function is called from the drivers IPC service
    #
    #    \param assetid: The ID of the service you wish to initialise"""
    def Spinup(self, assetid,context):
        self.logger.info('Spinup Request - Performing Restart')
        self.Restart()
        
        return    

    def LogBasic(self,msg):
        self.logger.info(msg)        

    def LogDetail(self,msg):
        self.logger.debug(msg)        

    ## Resume a paused service
    #
    #    This function is called from the drivers IPC service
    #
    #    \param assetid: The ID of the service you wish to pause """
    def Resume(self,assetid,context):
        for drv in self.drivers:
            if drv.asset == assetid and drv.context == context:
                drv.Resume()
        #self.drivers[assetid].Resume()

    ## Enable or disable debugging on a particular source.
    #
    #    This function is called from the drivers IPC service
    #
    #    \param assetid: The ID of the service you wish to debug
    #    \param action: '1' to enable debugging, '0' to disable it"""
    def DebugMode(self,assetid,action):
        pass
##        self.logger.info('Driver Debug Mode Request')
##        enable = True
##        try:
##            if (int(action) == 0):
##                enable = False
##        except:
##            pass
##        
##        log = logging.getLogger('Source '+ str(assetid))
##        if len(log.handlers) > 0:
##            if enable == True:
##                self.logger.info("Turning On Source Logging For Asset " + str(assetid))
##                log.handlers[0].setLevel(logging.DEBUG)
##                log.setLevel(logging.DEBUG)
##            else:
##                self.logger.info("Turning Off Source Logging For Asset " + str(assetid))
##                log.handlers[0].setLevel(logging.INFO)
##                log.setLevel(logging.INFO)
##        #else:
##        #    print "No Handlers - Log Level Not Changed"
##            
##        try:
##            #print dir(self.drivers[assetid].driver)
##            if (hasattr(self.drivers[assetid].driver,'DebugMode')):
##                self.logger.info("Running Source-Specific Debugging Routine")                
##                self.drivers[assetid].driver.DebugMode(enable)
##            else:
##                self.logger.info("This Source Does Not Have A Debugging Routine")              
##        except Exception as e:
##            self.logger.info("Unable To Set DebugMode On Source: " + str(e))
##            pass

    def ShowStatus(self):
        final = []
        for core in self.drivers:
            final.append(core.Status())
        return json.dumps(final)

    def ShowLog(self,coreid,context,logtype):        
        if coreid == 0:
            if logtype == "basic":                
                return "\n".join(reversed(self.basiclog))                
            else:
                return "\n".join(reversed(self.detaillog))
            
        for cr in self.drivers:
            #cr = self.drivers[core]            
            coreid = int(coreid)
            if int(cr.asset)==coreid:                
                if int(cr.context) == int(context):                    
                    if logtype == "basic":                
                        return "\n".join(reversed(cr.basiclog))
                    else:
                        return "\n".join(reversed(cr.detaillog))
        return ""

    def ShowIssues(self,coreid,context):
        if coreid == 0:
            return ""
        for cr in self.drivers:
            #cr = self.drivers[core]            
            if int(cr.asset)==coreid:
                if int(cr.context) == int(context):
                    try:
                        return "\n".join(cr.driver.issues)
                    except:
                        return "Driver Does Not Support 'Issues'"
        return ""

    def StartTracing(self,tracename):
        
        if tracename == "":
            self.logger.info("Tracing Disabled")
            self.tracing = None
            for c in self.drivers:
                c.tracing = None
                #self.drivers[c].tracing = None
        else:
            tracename = tracename.replace('~','|')
            self.logger.info("Beginning New Trace: " + str(tracename))
            self.tracing = tracename
            for c in self.drivers:
                #print("Setting Trace For " + c)
                c.tracing = tracename
                #self.drivers[c].tracing = tracename

    ## Alters the update rate for the data source
    #
    #   This function is called from the web interface
    #
    #    \param assetid: The ID of the service you wish to change
    #    \param rate: The rate at which to update the data source (in seconds) """
    def SetUpdateRate(self,assetid,rate,context):
        try:
            for cr in self.drivers:                       
                coreid = int(coreid)
                if int(cr.asset)==coreid:
                    if cr.context == context:
                        self.drivers[assetid].SetUpdateRate(float(rate))
        except:
            pass
