## @package ardi.driver.eventdrivercore
#  ARDI Event Driver Connection
#
#  This namespace contains the EventDriverCore, a class that represents a connection to a data source.
#
#import pycurl
import requests
import xmltodict
import time
import socket
import ardi.driver.transform as transform
import ardi.driver.logsystem as logsystem
import platform
from subprocess import Popen
import subprocess
import logging
import logging.handlers
try:
	import thread
except:
	import threading as thread
import os
import pytz
from tzlocal import get_localzone
import datetime
from datetime import timedelta
import traceback
import math

try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode
            
## Data Point Information
#
#  An empty class used for storing information about data points
class pointinfo:
    pass

## A single request for event data.
#
#  This object represents a single request for new event data.
#
#  It contains a number of properties as well as functions to make your data
#  consistent.
class eventquery:
    def __init__(self,engine,params):

        ## A reference to the Core
        self.core = engine
        ## The function that is being run - often 'interp'
        #self.function = function
        ## True if this query expects sub-second resolution
        self.subsecond = False        
       
        ## The UTC string for the end time
        try:
            self.ends = params["end"]
        except:
            self.ends = "2022-01-01 01:00:00"

        ## The UTC string for the start time
        try:
            self.starts = params["start"]
        except:
            self.starts = "2022-01-01 00:00:00"

        #print(str(params))
        start = self.starts
        end = self.ends
        self.maxcount = 1000
        self.offset = 0

        if 'count' in params:
            self.maxcount = float(params['count'])    
        if 'offset' in params:
            self.offset = float(params['offset']) 
        
        ## An array of the I/O points to be returned
        if "search" in params:
            self.search = params["search"]

        ## The timezone for most dates
        self.tz = "UTC"        

        ## Contains the content that will be returned.
        self.output = ""
        ## A set of samples ready for processing
        self.outset = []        

        ## True if the driver handles the interpolation for us, otherwise False.        
        self.initialised = False

        self.addrbindings = {}
        self.addresses = []        

        self.localtz = datetime.datetime.now().tzinfo        

        ## The database-compliant start date
        try:
            self.sd = datetime.datetime.strptime( start, "%Y-%m-%d %H:%M:%S" )
        except:
            self.sd = start

        ## The database-compliant end date
        try:
            self.ed = datetime.datetime.strptime( end, "%Y-%m-%d %H:%M:%S" )
        except:
            self.ed = end

        #self.span = (self.ed - self.sd).total_seconds()             

        ## A UTC datetime for the start date
        self.localstart = self.sd.replace(tzinfo=pytz.utc).astimezone(self.localtz)
        ## A UTC datetime for the end date
        self.localend = self.ed.replace(tzinfo=pytz.utc).astimezone(self.localtz)

        self.outset = []
        self.outtimes = []
        

    ## Finishes output
    #
    #  Some functions need to add additional items when processing is complete.
    #
    def Cleanup(self):
        #Sort and trim the output events...
        self.outtimes.sort(key=lambda x: x[0],reverse=True)
        final = []
        counter = 0
        for x in self.outtimes:            
            if self.offset > 0:                
                self.offset -= 1
                continue

            final.append(self.outset[x[1]])

            counter += 1
            if counter > self.maxcount:
                break

        self.outtimes = []
        self.outset = final

    ## Called when all lines are added
    #
    #  This function finishes off the request and returns the values that should
    #  be sent back to the caller.
    def Finish(self):
        self.core.logger.info("Query Complete")
        
        self.Cleanup()
        output = "\r\n".join(self.outset)
        
        return output.encode('ascii')

    ## Prepare the Request
    #
    #  This function prepares the internal storage for special functions, such as
    #  interpolation, min/max or standard deviation.
    def Initialise(self):
        pass

    ## Add a line of output
    #
    #  This adds a single address/timestamp/value record.
    #
    #  Note that this isn't always output literally - depending on which function you are using,
    #  this data might be summarised or even ignored completely.
    #
    # @param addr The address of the incoming data (for example, tag name).
    # @param stamp A database-native timestamp
    # @param value The value
    def AddLine(self,name,start,end,additional=None):
        self.outtimes.append((start,len(self.outset)))
        self.outset.append((name + "," + start + "," + end).strip())           

    def AddLineFromDictionary(self,dct):
        nm = ""
        strt = ""
        end = ""

        if 'name' in dct:
            nm = dct['name']
        if 'message' in dct:
            nm = dct['message']
        if 'start' in dct:
            strt = dct['start']
        if 'end' in dct:
            end = dct['end']

        if nm != "":
            self.AddLine(nm,strt,end)

## A connection to a single instance or device
#
#    Unlike live drivers, event drivers are mostly passive and don't run their own individual threads.
#
#    They wait until they are queried by the system to return data.
class eventcore:
    
    ## Initialises the Core
    def __init__(self,driver):
        
        ##The url to return data to, used for remote drivers
        self.url = "localhost"

        ##The source ID
        self.asset = 1

        ##The actual driver instance - the class that performs the query on the data store
        self.driver = driver

        ##A self-reference        
        driver.core = self

        ##A number to track any dynamic addresses (usually associated with random data)
        driver.dynaddress = 0

        ##Internal Log Lists
        self.basiclog = []
        self.detaillog = []
        self.issues = []

        self.qry = eventquery(self,{})

    ## Set Connection Details
    #
    #    This function sets some of the basic identity information about the core.
    #
    #    Args:
    #        url: The URL to query to get core configuration information from ARDI
    #        assetid: The source ID that this core represents"""   
    def SetARDIDetails(self, url, assetid):
         
        self.url = url
        self.asset = assetid
        self.InitLogger()

    ## Set Driver
    #
    #    This function associates the given driver with this core
    #
    #    Args:
    #        driver: Associate this core with a driver"""    
    def SetARDIDriver(self, driver):   
        self.driver = driver
        self.driver.asset = self.asset
        driver.core = self

    ## Closes to connection to the data store
    #
    # This function is largely unnessicary - most query drivers don't actually connect until query-time.
    def Close(self):
        self.driver.Disconnect()

    ## Initialises the logging for this indiviual source
    def InitLogger(self):

        if hasattr(self,'logger'):
            return
        
        #print 'Initialising Logger for ' + self.asset
        logfile = '/var/log/ardi/source-' + self.asset + '-' + self.profile + '-event.log'
        if (platform.system() == 'Windows'):
        	try:
        		logfolder = os.environ['ARDILogPath']
        	except:
        		logfolder = "c:\\windows\\temp"
        	logfile = logfolder + "\\ardi-source-" + self.asset + '-' + self.profile + '-event.log'

        ##The logger for a specific data source, rather than the driver process as a whole
        self.logger = logsystem.CreateLogger(self,'Source '+self.asset,logfile)
            
        self.logger.info('----SOURCE STARTED----')
        self.logger.info('Starting Event Data Source ' + self.asset + " on thread " + str(thread.get_ident()))

    ## Force the core to reload
    #
    #    This function is usually called after a change to the required data
    #
    #    It is needed to update the translation tables and transforms that may apply.
    def Reload(self):      
        self.LoadConfig()

    ## Reload driver configuration

    # This function triggers the driver to reload it's details from the ARDI server. Note that this
    #   is very different from the same function located in the DriverBase class. The data returned by
    #    this query includes all of the individual data points to be sampled from the data source.
    def LoadConfig(self):
        
        if (self.source != ''):
            fl = open(self.source,'r')
            xml = fl.read()
            fl.close()
            self.parseconfig(xml)
            return
            
        #response = StringIO()
        
        fullurl = "http://" + self.url + "/api/sourcedata?asset=" + str(self.asset) + "&mode=4&profile="+str(self.profile)

        self.logger.debug("Getting Driver Config From " + fullurl)
        
        #c = pycurl.Curl()
        #c.setopt(c.URL,fullurl)
        #c.setopt(c.WRITEFUNCTION,response.write)
        #c.setopt(c.TIMEOUT,5)
        #c.setopt(c.CONNECTTIMEOUT,5)
        #c.perform()
        #c.close()
        web = requests.get(fullurl,timeout=30)
        
        self.parseconfig(web.text)

    ## Parse the configuration details
    #
    #    This function processes the XML from 'LoadConfig' to setup the driver and the many data points."""
    def parseconfig(self,xml):
        #print xml
        result = xmltodict.parse(xml)
        
        #print str(result)
        
        #Core Parameters
        self.driveraddr = result['sourcedata']['connection']['address']

        self.hostport = int(result['sourcedata']['connection']['port'])
        self.driver.SetAddress(self.driveraddr)

        if self.driveraddr is None:
            self.driveraddr = ""

        try:
            self.driver.Remap()
        except:
            pass

    ## Run a Query
    #
    #   This function is used to actually query the data source.
    #
    #   Modern drivers should provide a 'Query' function - this will be passed a eventquery instance.
    #
    #    \param points: A list of points to query
    #    \param start: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param end: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param function: The function to perform. Functions include 'raw', 'avg', 'sum', 'min' and 'max'
    #    \param grain: See the documentation on 'grain' in historianquery2
    #    \param first: True if this the data is to include the samples immediately preceeding the start time.
    def Query(self, instance,parameters):     
       
        if "start" in parameters:
            if parameters["start"] == "now":
                parameters["start"] = datetime.datetime.now()
        try:
            #print str(dir(self.driver))
            hq = eventquery(self,parameters)
            return self.driver.Query(hq)
        except:
            self.logger.exception("Event Query Failed")
            return None               

    ## Gets the status of this source as JSON
    #
    #    This function returns basic status information as a dictionary."""
    def Status(self):
        value = {}        
        value['asset'] = self.asset        
        if self.driver.connected == True:
                value['connected'] = 1
        else:
                value['connected'] = 0
        try:
                value['issues'] = len(self.driver.issues)
        except:
                value['issues'] = 0

        try:
                value['context'] = len(self.profile)
        except:
                value['context'] = 0
        
        return value

    ## Starts the driver
    #
    # This function initiates the driver.
    #
    # @param address The encoded destination address (ie. IP address) of the device
    # @param asset The asset id that this source represents"""
    def Launch(self,address,asset):
        
        self.SetARDIDetails(address,asset)
        self.LoadConfig()    
        self.driver.Connect()
                    
