## @package ardi.driver.histdrivercore
#  ARDI Historian Driver Connection
#
#  This namespace contains the HistDriverCore, a class that represents a connection to a data source.
#
#import pycurl
import requests
import xmltodict
import time
import socket
import ardi.driver.transform as transform
import ardi.driver.logsystem as logsystem
import platform
from subprocess import Popen
import subprocess
import logging
import logging.handlers
try:
	import thread
except:
	import threading as thread
import os
import pytz
from tzlocal import get_localzone
import datetime
from datetime import timedelta
import traceback
import math

try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode
            
## Data Point Information
#
#  An empty class used for storing information about data points
class pointinfo:
    pass

## A single request for historical data.
#
#  This object represents a single request for new historical data.
#
#  It contains a number of properties as well as functions to make your data
#  consistent.
class historianquery2:
    def __init__(self,engine,points,start,end,function,grain,first):

        ## A reference to the Core
        self.core = engine
        ## The function that is being run - often 'interp'
        self.function = function
        ## True if this query expects sub-second resolution
        self.subsecond = False
        ## Internal storage of recent values for change-queries
        self.lastvalues = {}
        ## Internal storage of recent timestamps for change-queries
        self.lasttimestamps = {}
        ## The expected resolution.
        #
        # A positive number for grain indicates the number of seconds between each expected sample (ie. 20s, 60s etc.)
        #
        # A negative number for grain indicates the number of slices to break the range into - ie, a 60 minute range with a grain of -60 should be broken into 1-minute slices.
        self.grain = grain
        ## The UTC string for the end time
        self.ends = end
        ## The UTC string for the start time
        self.starts = start
        ## TRUE if this is the first query, FALSE if it's a secondary query (ie. time is being played and we already have the first batch of information)
        self.first = first
        ## An array of the I/O points to be returned
        self.points = points
        ## The timezone for most dates
        self.tz = "UTC"
        ## The style of the dates.
        #
        #  1 is SQL-compliant timestamps.
        #  2 is Unix-style EPOCH.
        #  3 is Windows FILETIME
        #  4 is .NET EPOCH
        self.datestyle = "1"

        ## Contains the content that will be returned.
        self.output = ""
        ## A set of samples ready for processing
        self.outset = []
        ## TRUE if date components should be treated as strings, otherwise FALSE.
        self.quotestring = False

        ## True if the driver handles the interpolation for us, otherwise False.
        self.interpolates = False
        self.initialised = False

        self.addrbindings = {}
        self.addresses = []

        self.calclocally = False

        self.localtz = pytz.UTC

        self.customDTToNative = str
        self.customNativeToDT = str
        self.customNativeToDTLocal = str

        ## The database-compliant start date
        self.sd = datetime.datetime.strptime( start, "%Y-%m-%d %H:%M:%S" )
        ## The database-compliant end date
        self.ed = datetime.datetime.strptime( end, "%Y-%m-%d %H:%M:%S" )

        self.span = (self.ed - self.sd).total_seconds()

        grn = int(self.grain)
        if grn == 0:
            grn = 1
        if grn < 0:
            ## The size of the 'window' that should cover each returned sample.
            self.window = self.span / -grn
        else:
            self.window = grn

        ## A UTC datetime for the start date
        self.localstart = self.sd
        ## A UTC datetime for the end date
        self.localend = self.ed

        if self.function == "min" or self.function == "max":
            #self.core.logger.info("Performing Min/Max Calc")
            self.calclocally = True
            try:
                if self.core.Supports(self.function) == True:
                    self.calclocally = False
                    #self.core.logger.info("Driver supports calculation")
            except:
                self.core.logger.info("Handling Min/Max Locally")
                self.calclocally = True                
                
            self.calchannels = []            
        if self.function == "avg":
            self.core.logger.info("Performing Avg Calc")
            self.calclocally = True
            try:
                if self.core.Supports(self.function) == True:
                    self.calclocally = False
                    self.core.logger.info("Driver supports calculation")
            except:
                self.core.logger.info("Driver doesn't support calc - processing locally")
                self.calclocally = True
                #self.grain = 1
                
            self.totals = []
            self.counters = []

        if self.ed == self.sd:
            self.touched = []

        self.buffer = {}
        self.buffercount = {}

        qryindx = -1
        for pnt in points:
            qryindx = qryindx+1
            plindex = -1
            for itm in self.core.pointlist:
                plindex = plindex + 1
                if itm.code == pnt:
                    #print itm.code + " = " + pnt + "..."
                    if itm.address not in self.addresses:
                        self.addresses.append(itm.address)
                        self.addrbindings[itm.address] = []
                    self.addrbindings[itm.address].append((qryindx,plindex))
                    if self.function == "min":
                        self.calchannels.append(999999.9)
                        #self.calchannels[plindex] = 999999.9
                    if self.function == "max":
                        self.calchannels.append(-999999.9)
                        #self.calchannels[plindex] = -999999.9
                    if self.function == "avg":
                        #try:
                        self.counters.append(0)
                        self.totals.append(0)
                        #except:
                        #    self.counters.add(plindex,0)
                        #    self.totals.add(plindex,0)

    ## Finishes output
    #
    #  Some functions need to add additional items when processing is complete.
    #
    def Cleanup(self):        
        stamp = datetime.datetime.strptime( self.ends, "%Y-%m-%d %H:%M:%S" )
        if (self.function == "changes"):
            #In 'Changes' mode, we need to output any resudual values...
            for keys in self.lasttimestamps:
                if self.lasttimestamps[keys] != False:
                    vv = str(keys) + "," + str(stamp) + "," + str(self.lastvalues[keys])# + "\r\n"
                    self.outset.append(vv)        

    ## Converts a local date-time to database-native format.
    def DTTextToNative(self,dt,tzoverride = ""):
        #Traditional date...
        #self.core.logger.info("Converting " + str(dt) + " To Native")
        tz = self.tz
        localtz = self.localtz

        if (tzoverride != ""):
            tz = tzoverride
            if tz == "local":
                localtz = get_localzone()
            else:
                if tz == "UTC":
                    localtz = pytz.UTC
                else:
                    localtz = pytz.timezone(self.tz)

        if self.datestyle == "1":
            if tz != "UTC":
                dt = localtz.localize(dt)
                dt = dt.astimezone(pytz.UTC).replace(tzinfo=None)
            if self.quotestring==True:
                return "'" + str(dt) + "'";
            else:
                return str(dt);

        #UNIX Epoch
        if self.datestyle == "2":
            if tz != "UTC":
                dt = localtz.localize(dt)
                dt = dt.astimezone(pytz.UTC)
            return str(long((dt - datetime.datetime(1970,1,1,0,0,0,0,pytz.UTC)).total_seconds()))

        #Win32 FILETIME
        if self.datestyle == "3":
            tickspersec = 10000000
            if self.tz != "UTC":
                dt = localtz.localize(dt)
                dt = dt.astimezone(pytz.UTC)
            l = long((dt - datetime.datetime(1601,1,1,0,0,0,0,pytz.UTC)).total_seconds())
            return str(l * tickspersec);
        
        #CLR Time
        if self.datestyle == "4":
            tickspersec = 10000000
            if self.tz != "UTC":
                dt = localtz.localize(dt)
                dt = dt.astimezone(pytz.UTC)
            diff = (dt - datetime.datetime(1,1,1,0,0,0,0,pytz.UTC)).total_seconds()
            return str(long(tickspersec) * long(diff));

    ## Converts a UTC date-time to database-native format.
    def DTUTCToNative(self,dt,tzoverride = ""):
        #Traditional date...
        #self.core.logger.info("Converting " + str(dt) + " To Native")
        tz = self.tz
        localtz = self.localtz

        if (tzoverride != ""):
            tz = tzoverride
            if tz == "local":
                localtz = get_localzone()
            else:
                if tz == "UTC":
                    localtz = pytz.UTC
                else:
                    localtz = pytz.timezone(self.tz)

        if tz != "UTC":
            dt = pytz.UTC.localize(dt)
            dt = dt.astimezone(localtz).replace(tzinfo=None)

        if self.datestyle == "2":
            if tz != "UTC":
                dt = localtz.localize(dt)
                dt = dt.astimezone(pytz.UTC)
            return str(long((dt - datetime.datetime(1970,1,1,0,0,0,0,pytz.UTC)).total_seconds()))

        #Win32 FILETIME
        if self.datestyle == "3":
            tickspersec = 10000000
            if self.tz != "UTC":
                dt = localtz.localize(dt)
                dt = dt.astimezone(pytz.UTC)
            l = long((dt - datetime.datetime(1601,1,1,0,0,0,0,pytz.UTC)).total_seconds())
            return str(l * tickspersec);
        
        #CLR Time
        if self.datestyle == "4":
            tickspersec = 10000000
            if self.tz != "UTC":
                dt = localtz.localize(dt)
                dt = dt.astimezone(pytz.UTC)
            diff = (dt - datetime.datetime(1,1,1,0,0,0,0,pytz.UTC)).total_seconds()
            return str(long(tickspersec) * long(diff));

        #64Bit Epoch Time (Milliseconds)
        if self.datestyle == "5":
                if tz != "UTC":
                        dt = localtz.localize(dt/10000000)
                        dt = dt.astimezone(pytz.UTC)
                return str(long((dt - datetime.datetime(1970,1,1,0,0,0,0,pytz.UTC)).total_seconds()))

        #Custom DT Function
        if self.datestyle == "6":
                return self.customDTToNative(dt)

        if self.quotestring==True:
            return "'" + str(dt) + "'";
        else:
            return str(dt);

    ## Converts database-native date to a local one.
    def DTNativeToLocalDate(self,dt):
        #print 'Got ' + str(dt)
        #Traditional date...
        if self.datestyle == "1":
            return dt
    
        #UNIX Epoch
        if self.datestyle == "2":
            dd = datetime.datetime.utcfromtimestamp(long(dt)).replace(tzinfo=pytz.UTC)
            return dd.astimezone(self.outputtz).replace(tzinfo=None)
        
        #Win32 FILETIME
        if self.datestyle == "3":
            tickspersec = 10000000
            l = long(dt) / tickspersec
            dvx = (datetime.datetime(1970,1,1,0,0,0,0,pytz.UTC) - datetime.datetime(1601,1,1,0,0,0,0,pytz.UTC)).total_seconds()
            l -= dvx
            #print "Final Timestamp: " + str(l)
            dd = datetime.datetime.utcfromtimestamp(l).replace(tzinfo=pytz.UTC)
            return dd.astimezone(self.localtz).replace(tzinfo=None)

        #CLR Time
        if self.datestyle == "4":
            tickspersec = 10000000
            diff = long(dt) / tickspersec
            tstamp = datetime.datetime(1970,1,1,0,0,0,0,pytz.UTC)
            dvx = (datetime.datetime(1970,1,1,0,0,0) - datetime.datetime(1,1,1,0,0,0)).total_seconds()
            #print "Total Seconds: " + str(long(dvx))
            dd = datetime.datetime.utcfromtimestamp(diff - dvx).replace(tzinfo=pytz.UTC)
            return dd.astimezone(self.localtz).replace(tzinfo=None)

        #64Bit Epoch Time (Milliseconds)
        if self.datestyle == "5":
                classic = long(long(dt)/10000000)
                micro = long(dt) % 10000000
                dd = datetime.datetime.fromtimestamp(classic)
                dd = dd.replace(microsecond=micro)
                return dd.replace(tzinfo=None)

        #Custom DT Function
        if self.datestyle == "6":
                return self.customNativeToDTLocal(dt)

    ## Converts a database-native date to a UTC one.
    def DTNativeToUTCDate(self,dt):
        #Traditional date...
        #print("Incoming: " + str(dt))
        if self.datestyle == "1":
            try:
                if self.tz != "UTC":
                    dt = self.outputtz.localize(dt)
                    #print "Ummm... localised, its " + str(dt)
                    dt = dt.astimezone(pytz.UTC).replace(tzinfo=None)
                    #print "Final Result: " + str(dt)
                return dt
            except:
                ps = dt.find('.')
                if ps > -1:                    
                    dt = dt[0:ps]
                dt = datetime.datetime.strptime( dt, "%Y-%m-%d %H:%M:%S" )
                if self.tz != "UTC":
                    dt = self.outputtz.localize(dt)
                    #print "Ummm... lcalised, its " + str(dt)
                    dt = dt.astimezone(pytz.UTC).replace(tzinfo=None)
                    #print "Final Result: " + str(dt)
            return dt

        #UNIX Epoch
        if self.datestyle == "2":
            dd = datetime.datetime.utcfromtimestamp(long(dt)).replace(tzinfo=pytz.UTC)
            return dd.replace(tzinfo=None)

        #Win32 FILETIME
        if self.datestyle == "3":
            tickspersec = 10000000
            l = long(dt) / tickspersec
            dvx = (datetime.datetime(1970,1,1,0,0,0,0,pytz.UTC) - datetime.datetime(1601,1,1,0,0,0,0,pytz.UTC)).total_seconds()
            l -= dvx
            #print "Final Timestamp: " + str(l)
            dd = datetime.datetime.utcfromtimestamp(l).replace(tzinfo=pytz.UTC)
            return dd.replace(tzinfo=None)

        #CLR Time
        if self.datestyle == "4":
            tickspersec = 10000000
            diff = long(dt) / tickspersec
            tstamp = datetime.datetime(1970,1,1,0,0,0,0,pytz.UTC)
            dvx = (datetime.datetime(1970,1,1,0,0,0) - datetime.datetime(1,1,1,0,0,0)).total_seconds()
            #print "Total Seconds: " + str(long(dvx))
            dd = datetime.datetime.utcfromtimestamp(diff - dvx).replace(tzinfo=pytz.UTC)
            return dd.replace(tzinfo=None)

        #64Bit Epoch Time (Milliseconds)
        if self.datestyle == "5":
                classic = long(long(dt)/10000000)
                micro = long(dt) % 10000000
                dd = datetime.datetime.utcfromtimestamp(classic).replace(tzinfo=pytz.UTC)
                dd = dd.replace(microsecond=micro)
                return dd.replace(tzinfo=None)

        #Custom DT Function
        if self.datestyle == "6":
                return self.customNativeToDT(dt)

    ## Sets the date format and timezones for this query.
    #
    # @param timezone The name of the timezone to use
    # @param style The style of date-stamp in use
    # @param quotestrings Interpret the date stamps as strings when writing them.
    def SetDateFormat(self,timezone,style=1,quotestrings=False):
        self.datestyle = style
        self.tz = timezone
        self.tz_out = timezone
        if self.tz == "local":
            self.localtz = get_localzone()
            self.outputtz = get_localzone()
        else:
            if self.tz == "UTC":
                self.localtz = pytz.UTC
                self.outputtz = pytz.UTC
            else:
                self.localtz = pytz.timezone(self.tz)
                self.outputtz = pytz.timezone(self.tz)

        self.quotestring = quotestrings
        self.sd = self.DTUTCToNative(self.sd)
        self.ed = self.DTUTCToNative(self.ed)
        #self.core.logger.info("Range: " + str(self.sd) + " - " + str(self.ed) + " ( UTC - From " + timezone + " )")

    ## Set date formats and timezones for input and output.
    #
    #  Some databases (ie. Citect) are asymetrical - they require the request in one timezone
    #  but return the times in another.
    #
    #  @param timezonei The timezone that incoming / filter data comes in as
    #  @param timezoneo The timezone that data coming from the database uses
    #  @param style The style of date in the database
    #  @param quotestrings TRUE if the filter dates should be treated as strings in a query.
    def SetDateFormatIO(self,timezonei,timezoneo,style=1,quotestrings=False):
        self.datestyle = style
        self.tz = timezonei
        self.tz_out = timezoneo
        if self.tz == "local":
            self.localtz = get_localzone()
        else:
            if self.tz == "UTC":
                self.localtz = pytz.UTC
            else:
                self.localtz = pytz.timezone(self.tz)
                
        if self.tz_out == "local":
            self.outputtz = get_localzone()
        else:
            if self.tz_out == "UTC":
                self.outputtz = pytz.UTC
            else:
                self.outputtz = pytz.timezone(self.tz)

        self.quotestring = quotestrings
        self.sd = self.DTUTCToNative(self.sd)
        self.ed = self.DTUTCToNative(self.ed)
        #self.core.logger.info("Range: " + str(self.sd) + " - " + str(self.ed) + " ( UTC - From " + timezonei + " )")

    ## Called when all lines are added
    #
    #  This function finishes off the request and returns the values that should
    #  be sent back to the caller.
    def Finish(self):
        #self.core.logger.info("Query Complete")
        output = ""

        if self.function == "interp" or self.function == "continue":
            if self.interpolates == False:
                lines = []
                for nm in self.buffer:
                    #nm = n.code
                    for k,v in self.buffer[nm].items():
                        try:
                            vl = v / self.buffercount[nm][k]
                        except:
                            vl = v
                        stamp = self.localstart + datetime.timedelta(seconds=k*self.window)
                        lines.append(nm + "," + stamp.strftime( "%Y-%m-%d %H:%M:%S" ) + "," + str(vl))
                        #output += nm + "," + stamp.strftime( "%Y-%m-%d %H:%M:%S" ) + "," + str(vl)# + "\r\n"

                output = "\r\n".join(lines)
                return output.encode('ascii')

        #Return the results of local calculations (min/max/avg)
        if self.calclocally == True:
                   
            if self.function == "min" or self.function == "max":
                lines = []
                for nm in self.buffer:
                    #nm = n.code
                    for k,v in self.buffer[nm].items():
                        vl = v
                        stamp = self.localstart + datetime.timedelta(seconds=k*self.window)
                        lines.append(nm + "," + stamp.strftime( "%Y-%m-%d %H:%M:%S" ) + "," + str(vl))

                output = "\r\n".join(lines)
                return output.encode('ascii')

            if self.function == "avg":
                output = ""
                dt = self.localstart.strftime( "%Y-%m-%d %H:%M:%S" )
                cnt = -1
                for chan in self.totals:
                    cnt = cnt + 1
                    count = self.counters[cnt]
                    if count > 0:
                        count = chan / count
                    output += str(cnt) + "," + dt + "," + str(count)# + "\r\n"
                return output.encode('ascii')

            
        self.Cleanup()
        output = "\r\n".join(self.outset)
        #for line in self.outset:
        #    output += line
        
        #print "Output: " + output
        return output.encode('ascii')

    ## Prepare the Request
    #
    #  This function prepares the internal storage for special functions, such as
    #  interpolation, min/max or standard deviation.
    def Initialise(self):
        if self.interpolates == False:
            self.buffer = {}
            self.buffercount = {}
        if self.calclocally == False:
            self.buffer = {}
            self.buffercount = {}
            self.records = []

    ## Add a line of output
    #
    #  This adds a single address/timestamp/value record.
    #
    #  Note that this isn't always output literally - depending on which function you are using,
    #  this data might be summarised or even ignored completely.
    #
    # @param addr The address of the incoming data (for example, tag name).
    # @param stamp A database-native timestamp
    # @param value The value
    def AddLine(self,addr,stamp,value):
        if value is None:
            return

        if self.initialised == False:
            self.Initialise()
            self.initialised = True

        vfunc = self.function
        if self.calclocally == False:
            if (self.function == "min") or (self.function=="max") or (self.function=="avg"):
                vfunc = "interp"
        
        #Convert the stamp to UTC and output it.
        stamp = self.DTNativeToUTCDate(stamp)
        try:
            #print "Handling..." + str(addr) + " / " + str(stamp) + " / " + value
            for bind in self.addrbindings[addr]:
                vx = value
                #print addr + " Data Is Bound To " + self.core.pointlist[bind[1]].code
                id = bind[0]
                #Apply any transformations or corrections to the data here.
                if self.core.pointlist[bind[1]].transform != "":
                    tx = transform.transform(self.core.pointlist[bind[1]].transform)
                    #print "Transforming: " + self.core.pointlist[bind[1]].transform
                    vx = tx.run(vx)

                if self.core.pointlist[bind[1]].discrete == True:
                    #print(self.core.pointlist[bind[1]].code + " is discrete!")
                    try:
                        vx = int(vx)
                    except:
                        pass

                if self.core.pointlist[bind[1]].lookupid != 0:
                    try:
                        lid = self.core.pointlist[bind[1]].lookupid
                        vx = self.core.base.lookups[lid][str(vx)]
                    except:
                        vx = "None / Unknown"

                #Only perform aggregation on CONTINUOUS points...
                if self.core.pointlist[bind[1]].discrete == False:

                    #This section handles interpolation for sources that don't support grain.
                    if vfunc == "interp" or vfunc == "continue":
                        if self.interpolates == False:

                            indxno = 0
                            try:
                                indxno = int(math.floor((stamp - self.localstart).total_seconds() / self.window))
                            except:
                                pass

                            nm = str(id)

                            #print "Dropping..."

                            if nm not in self.buffer:
                                self.buffer[nm] = {}
                                self.buffercount[nm] = {}                        
                            
                            try:
                                self.buffer[nm][indxno] += float(vx)
                                self.buffercount[nm][indxno] += 1
                            except:
                                self.buffer[nm][indxno] = float(vx)
                                self.buffercount[nm][indxno] = 1
                            return

                    #This section handles any requests that are only looking for changes in your data
                    if (vfunc == "changes"):
                        nm = self.core.pointlist[bind[1]]
                        if nm not in self.lastvalues:
                            self.lastvalues[nm] = vx
                            self.lasttimestamps[nm] = False
                        else:
                            if (self.lastvalues[nm] == vx):
                                self.lasttimestamps[nm] = stamp
                                return
                            else:
                                oldstamp = self.lasttimestamps[nm]
                                self.lasttimestamps[nm] = False
                                oldvalue = self.lastvalues[nm]
                                self.lastvalues[nm] = vx
                                
                                if oldstamp != False:
                                    #print "Changed from " + str(oldvalue) + " to " + str(vx)
                                    self.outset.append(str(id) + "," + str(oldstamp) + "," + str(oldvalue))# + "\r\n")
                                    self.outset.append(str(id) + "," + str(stamp) + "," + str(vx))# + "\r\n")
                                    #self.output += str(id) + "," + str(oldstamp) + "," + str(oldvalue) + "\r\n" + str(id) + "," + str(stamp) + "," + str(vx) + "\r\n"
                                    return
                            
                    if vfunc == "min":
                            indxno = 0
                            try:
                                indxno = int(math.floor((stamp - self.localstart).total_seconds() / self.window))
                            except:
                                pass

                            nm = str(id)

                            if nm not in self.buffer:
                                self.buffer[nm] = {}
                                self.buffercount[nm] = {}                        
                            
                            try:
                                if self.buffer[nm][indxno] > float(vx):
                                        self.buffer[nm][indxno] = float(vx)                            
                            except:
                                self.buffer[nm][indxno] = float(vx)                        
                            
                            return
                    if vfunc == "max":                    
                            indxno = 0
                            try:
                                indxno = int(math.floor((stamp - self.localstart).total_seconds() / self.window))
                            except:
                                pass

                            nm = str(id)

                            if nm not in self.buffer:
                                self.buffer[nm] = {}
                                self.buffercount[nm] = {}                        
                            
                            try:
                                if self.buffer[nm][indxno] < float(vx):
                                        self.buffer[nm][indxno] = float(vx)                            
                            except:
                                self.buffer[nm][indxno] = float(vx)
                            return

                if self.ed == self.sd:
                    if str(id) in self.touched:
                        continue
                    self.touched.append(str(id))
                    
                self.outset.append(str(id) + "," + str(stamp) + "," + str(vx))# + "\r\n")                
        except:
            #traceback.print_exc()
            return

## Deprecated Version of the Historian Query
#
#  This earlier version of the HistorianQuery class was used in the Query() function.
#
#  @deprecated This class has been deprecated in favour of historianquery2 and the RunQuery() method.
class historianquery:
    def __init__(self,cr,fun,endtime):
        self.core = cr
        self.function = fun
        self.lastvalues = {}
        self.lasttimestamps = {}
        self.grain = 2
        self.ends = endtime

    def Cleanup(self,stamp):
        vv = ""
        if (self.function == "changes"):
            #In 'Changes' mode, we need to output any resudual values...
            for keys in self.lasttimestamps:
                if self.lasttimestamps[keys] != False:
                    vv += str(keys) + "," + str(stamp) + "," + str(self.lastvalues[keys])# + "\r\n"

        return vv

    def AddRecord(self, id, stamp, value, addr):
        if self.core.transforms[addr] != "":
            tx = transform.transform(self.core.transforms[addr])
            value = tx.run(value)

        if (self.function == "changes"):
            if id not in self.lastvalues:
                self.lastvalues[id] = value
                self.lasttimestamps[id] = False
            else:
                #print "Comparing Against " + str(value)
                if (self.lastvalues[id] == value):
                    self.lasttimestamps[id] = stamp
                    #print "Skipping Line!"
                    return ""
                else:   
                    oldstamp = self.lasttimestamps[id]
                    self.lasttimestamps[id] = False
                    oldvalue = self.lastvalues[id]
                    self.lastvalues[id] = value
                        
                    if oldstamp != False:
                        return str(id) + "," + str(oldstamp) + "," + str(oldvalue) + "\r\n" + str(id) + "," + str(stamp) + "," + str(value) + "\r\n"

        return str(id) + "," + str(stamp) + "," + str(value) + "\r\n"

## A connection to a single instance or device
#
#    Unlike live drivers, historian drivers are mostly passive and don't run their own individual threads.
#
#    They wait until they are queried by the system to return data.
class historiancore:
    
    ## Initialises the Core
    def __init__(self,driver):
        
        ##The url to return data to, used for remote drivers
        self.url = "localhost"

        ##The source ID
        self.asset = 1

        ##The actual driver instance - the class that performs the query on the data store
        self.driver = driver

        ##A self-reference        
        driver.core = self

        ##A number to track any dynamic addresses (usually associated with random data)
        driver.dynaddress = 0

        ##Internal Log Lists
        self.basiclog = []
        self.detaillog = []
        self.issues = []

        self.qry = historianquery(self,'raw',"")

    ## Set Connection Details
    #
    #    This function sets some of the basic identity information about the core.
    #
    #    Args:
    #        url: The URL to query to get core configuration information from ARDI
    #        assetid: The source ID that this core represents"""   
    def SetARDIDetails(self, url, assetid):
         
        self.url = url
        self.asset = assetid
        self.InitLogger()

    ## Set Driver
    #
    #    This function associates the given driver with this core
    #
    #    Args:
    #        driver: Associate this core with a driver"""    
    def SetARDIDriver(self, driver):   
        self.driver = driver
        self.driver.asset = self.asset
        driver.core = self

    ## Closes to connection to the data store
    #
    # This function is largely unnessicary - most query drivers don't actually connect until query-time.
    def Close(self):
        self.driver.Disconnect()

    ## Initialises the logging for this indiviual source
    def InitLogger(self):

        if hasattr(self,'logger'):
            return
        
        #print 'Initialising Logger for ' + self.asset
        logfile = '/var/log/ardi/source-' + self.asset + '-' + self.profile + '-hist.log'
        if (platform.system() == 'Windows'):
        	try:
        		logfolder = os.environ['ARDILogPath']
        	except:
        		logfolder = "c:\\windows\\temp"
        	logfile = logfolder + "\\ardi-source-" + self.asset + '-' + self.profile + '-hist.log'

        ##The logger for a specific data source, rather than the driver process as a whole
        self.logger = logsystem.CreateLogger(self,'Source '+self.asset,logfile)        
            
        self.logger.info('----SOURCE STARTED----')
        self.logger.info('Starting Historian Data Source ' + self.asset + " on thread " + str(thread.get_ident()))

    ## Force the core to reload
    #
    #    This function is usually called after a change to the required data
    #
    #    It is needed to update the translation tables and transforms that may apply.
    def Reload(self):              
        self.logger.info("----SOURCE RELOADING----")
        self.LoadConfig()
        try:
                self.logger.info("Disconnecting from Source")
                self.driver.Disconnect()
        except:
                pass
        try:
                self.logger.info("Reconnecting to Source")
                self.driver.Connect()
        except:
                pass

    ## Reload driver configuration

    # This function triggers the driver to reload it's details from the ARDI server. Note that this
    #   is very different from the same function located in the DriverBase class. The data returned by
    #    this query includes all of the individual data points to be sampled from the data source.
    def LoadConfig(self):
        
        if (self.source != ''):
            fl = open(self.source,'r')
            xml = fl.read()
            fl.close()
            self.parseconfig(xml)
            return
            
        #response = StringIO()
        
        fullurl = "http://" + self.url + "/api/sourcedata?asset=" + str(self.asset) + "&mode=1&profile="+str(self.profile)

        self.logger.debug("Getting Driver Config From " + fullurl)
        
        #c = pycurl.Curl()
        #c.setopt(c.URL,fullurl)
        #c.setopt(c.WRITEFUNCTION,response.write)
        #c.setopt(c.TIMEOUT,5)
        #c.setopt(c.CONNECTTIMEOUT,5)
        #c.perform()
        #c.close()
        web = requests.get(fullurl,timeout=30)
        
        self.parseconfig(web.text)

    ## Parse the configuration details
    #
    #    This function processes the XML from 'LoadConfig' to setup the driver and the many data points."""
    def parseconfig(self,xml):
        #print xml
        result = xmltodict.parse(xml)
        
        #print str(result)
        
        #Core Parameters
        self.driveraddr = result['sourcedata']['connection']['address']

        self.hostport = int(result['sourcedata']['connection']['port'])
        self.driver.SetAddress(self.driveraddr)

        if self.driveraddr is None:
            self.driveraddr = ""

        #Data Points
        #try:
        if True:
            self.pointlist = []
            self.transforms = {}
            if result['sourcedata']['points'] is not None:
                pointset = result['sourcedata']['points']['point']
                if '@address' in pointset:
                    #There is a Single Point
                    pointset = [pointset]
                
                for point in pointset:
                    #Add Data Points to List
                    if point['@address'] == '':
                        point['@address'] = point['@code']

                    self.transforms[point['@address']] = point['@transform']
                    
                    newpoint = pointinfo()
                    newpoint.address = point['@address']
                    newpoint.code = point['@code']
                    newpoint.changed = False
                    newpoint.value = ""
                    newpoint.intvalue = ""
                    newpoint.transform = point['@transform']
                    newpoint.discrete = False
                    newpoint.lookupid = 0
                    try:
                        newpoint.lookupid = int(point['@lookup'])
                        newpoint.discrete = True
                    except:
                        pass
                    try:
                        vv = newpoint.code.index(':value')
                        newpoint.discrete = True
                    except:
                        pass
                    try:
                        vv = newpoint.code.index(':state')
                        newpoint.discrete = True
                    except:
                        pass
                    self.pointlist.append(newpoint)
        
        self.logger.info(str(len(self.transforms)) + " Points/Transforms Loaded")

        try:
            self.driver.Remap()
        except:
            pass

    ## Run a Query
    #
    #   This function is used to actually query the data source.
    #
    #   Modern drivers should provide a 'RunQuery' function - this will be passed a historianquery2 instance.
    #
    #    \param points: A list of points to query
    #    \param start: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param end: The start date for the query, in YYYY-MM-DD hh:mm:ss format
    #    \param function: The function to perform. Functions include 'raw', 'avg', 'sum', 'min' and 'max'
    #    \param grain: See the documentation on 'grain' in historianquery2
    #    \param first: True if this the data is to include the samples immediately preceeding the start time.
    def Query(self, points,start,end,function,grain,first):     

        if hasattr(self.driver, 'RunQuery') == False:
            self.logger.warning("Driver Using Old (v1) Calls")
        else:
            try:
                #print str(dir(self.driver))
                hq = historianquery2(self,points,start,end,function,grain,first)
                return self.driver.RunQuery(hq)
            except:
                self.logger.exception("Historical Query (v2) Failed")
                return None
        
        finalpoints = []
        for pnt in points:
            for itm in self.pointlist:              
                if itm.code == pnt:
                    if itm.address not in finalpoints:
                          finalpoints.append(itm.address)
                    break

        if len(finalpoints) == 0:
            return "ERROR: No valid point specified"
        try:
            return self.driver.Query(finalpoints,start,end,function,grain,first)
        except:
            self.logger.exception("Historical Query Failed")
            return None

    ## Gets the status of this source as JSON
    #
    #    This function returns basic status information as a dictionary."""
    def Status(self):
        value = {}        
        value['asset'] = self.asset        
        if self.driver.connected == True:
                value['connected'] = 1
        else:
                value['connected'] = 0
        try:
                value['issues'] = len(self.driver.issues)
        except:
                value['issues'] = 0

        try:
                value['context'] = len(self.profile)
        except:
                value['context'] = 0
        
        return value

    ## Deprecated Function
    #
    # @deprecated This function has been eliminated completely from the new model.
    def GetResponder(self,func,endtime):
        return historianquery(self,func,endtime)

    ## Deprecated Function
    #
    # @deprecated You now call the AddLine function in the historianquery2 object.
    def AddRecord(self, id, stamp, value,addr):
        return self.qry.AddRecord(id,stamp,value,addr)

    ## Starts the driver
    #
    # This function initiates the driver.
    #
    # @param address The encoded destination address (ie. IP address) of the device
    # @param asset The asset id that this source represents"""
    def Launch(self,address,asset):
        
        self.SetARDIDetails(address,asset)
        self.LoadConfig()    
        self.driver.Connect()

    ## Check Driver Function Support
    #
    # This function is used to ask the driver if it supports specific aggregation functions, such as MIN and MAX
    #
    # @param func The lowercase name of the function"""
    def Supports(self,func):
        try:
            vl = self.driver.Supports(func)
            if vl is None:
                return False
            else:
                return vl
        except:
            return False
                    
