import json
import traceback
import importlib.util
from twisted.web import server, resource
from twisted.internet import reactor
import traceback
import urllib
import threading
import re
import sys
import os

#Defines a key-value output path
class KVOutput:
    def __init__(self,name,configfile="output.json",modules="",ignore=[]):
        self.name = name
        self.configfile = configfile
        self.destinations = []
        self.webcontent = {}
        self.batching = None
        self.webhost = None
        self.modules = modules
        if modules == "":
            for q in sys.path:
                final = q + "/ardi/util/outputengine"                
                if os.path.exists(final):
                    modules = final
                    self.modules = final
                    break                                
        else:
            self.modules = modules
        self.values = {}
        self.ignore = ignore
        self.initialised = False
        if self.modules != "":
            self.modules += "/"
        self.Load(configfile)        

    #Load the output configuration
    def Load(self,configfile):
        try:
            fl = open(configfile,'r')
            content = fl.read()
            jsn = json.loads(content)
            for x in jsn:
                if x['method'] in self.ignore:
                    print("This application is not compatible with " + x['method'] + " output.")
                    continue
                self.AddDestination(x)
            fl.close()
        except:
            traceback.print_exc()
            pass

    #Load a destination module
    def LoadDestination(self,d):
        method = d['method']        
        library = 'oe_' + method
        try:
            spec = importlib.util.spec_from_file_location(library,self.modules + library+".py")
            module = importlib.util.module_from_spec(spec)
            spec.loader.exec_module(module)
        except:
            traceback.print_exc()                     

        try:
            return module.output(d)
        except:
            traceback.print_exc()
            print(str(dir(module)))
            return None

    #Add a destination
    def AddDestination(self,d):
        try:
            ob = self.LoadDestination(d)
            if ob is not None:

                ob.webendpoint = None
                try:            
                    ob.hierarchy = ob.Supports('hierarchy')
                except:
                    ob.hierarchy = False
                    pass
                
                try:            
                    ob.webendpoint = ob.Supports('html')
                    if ob.webendpoint == False:
                        ob.webendpoint = None

                    if ob.webendpoint is not None:                    
                        if 'port' in d:
                            ob.webport = d['port']
                        else:
                            ob.webport = 8199
                except:
                    ob.webendpoint = None
                    pass

                try:
                    ob.filter = d['filter']
                except:
                    ob.filter = None

                try:
                    ob.allow = d['allow']
                    if isinstance(d['allow'],str):
                        tags = d['allow'].split(' ')
                        ob.allow = tags
                except:
                    ob.allow = None

                try:
                    ob.deny = d['deny']
                    if isinstance(d['deny'],str):
                        tags = d['deny'].split(' ')
                        ob.deny = tags
                except:
                    ob.deny = None

                ob.name = self.name + " " + d['method'].upper() + " Output"

                self.destinations.append(ob)
        except:
            print("Failed to load destination: " + d['method'])
            traceback.print_exc()
            return False
        return True

    #Initialise the system (connect to any data stores)
    def Initialise(self):
        if self.initialised == True:
            return

        for d in self.destinations:
            try:
                d.Initialise()
            except:
                traceback.print_exc()
                pass

        #See if we need to launch the web server
        webportno = 0
        enablewebserver = False
        indx = -1
        for d in self.destinations:
            indx += 1            
            try:                
                if d.webendpoint is not None:                    
                    enablewebserver = True
                    self.webcontent[d.webendpoint] = d
                    webport = d.webport
            except:
                traceback.print_exc()
                

        if enablewebserver == True:
            print("Starting Web Services")            
            self.webhost = OutputEngineWS(webport,self)
            self.webhost.Start()
            pass

        self.initialised = True

    def StartBatch(self):
        self.batching = []

    def EndBatch(self):
        indx = -1
        mx = len(self.batching)-1

        batch = self.batching
        self.batching = None

        for b in self.batching:
            indx += 1
            if indx >= mx:
                self.Set(b[0],b[1],options=b[2])
            else:
                options = b[2]
                if options is None:
                    options = {"batched": True}
                else:
                    options['batched'] = True

                self.Set(b[0],b[1],options=options)
        
   
    #Set a value
    def Set(self,name,value,options=None,stripparent=False,force=False):
        if self.initialised == False:
            self.Initialise()            
        if force == False:
            try:
                if self.values[name] == value:
                    return
            except:
                pass

        self.values[name] = value
        if isinstance(value,str):
            if options is None:
                options = ["text"]
            else:
                if "text" not in options:
                    options.append("text")

        for d in self.destinations:
            try:
                if d.filter is not None:                    
                    if re.search(d.filter,name) is None:                        
                        return

                if d.allow is not None:                  
                    for q in d.allow:                        
                        if q not in options:                           
                            return

                if d.deny is not None:
                    for q in d.deny:
                        if q in options:
                            return

                tname = name.replace("\t","")
                if d.hierarchy == False:
                    if stripparent == True:
                        bits = name.split("/")
                        tname = "/".join(bits[1:])                        
                    tname = tname.replace("/"," ") 
                    
                if self.batching is not None:
                    self.batching.append([name,value,options])
                else:
                    d.Set(tname,value)                
            except:
                pass
    
    def Stop(self):
        for d in self.destinations:
            try:
                d.Close()
            except:
                traceback.print_exc()
                pass

        if self.webhost is not None:
            self.webhost.Stop()


class InfoServer(resource.Resource):
    isLeaf = True

    def Message(self,request,message,state,code):
        content = "{"
        content += '  "message": "' + str(message) + '", "state": "' + str(state) + '", "code": ' + str(code)
        content += "}"
        
        request.responseHeaders.addRawHeader('Content-Type', 'application/json')
        request.responseHeaders.addRawHeader('Access-Control-Allow-Origin','*')
        return content.encode('utf-8')    

    #Handle HTTP POST Request
    def render_POST(self, request):
        body = request.content.read()
        return self.render_GET(request,body)

    #Handle HTTP GET Request
    def render_GET(self, request,body = None):
        #print(str(request.uri))
        if request.uri == "/favicon.ico":
            request.setResponseCode(404)            
            return "Not Found.".encode('utf-8')
        
        model = self.model

        uri = request.uri
        if uri[0] == '/':           
            uri = uri[1:]
        
        bits = uri.decode().split('/')
        if bits[0] == '':
            bits = bits[1:]  

        if bits[0] in model.webcontent:
            request.responseHeaders.addRawHeader('Access-Control-Allow-Origin','*')
            try:
                content = model.webcontent[bits[0]].HTML(uri,request)                        
            except:
                traceback.print_exc()
                pass
            return content.encode('utf-8')        
        
        request.setResponseCode(404)            
        return "Not Found.".encode('utf-8')

class OutputEngineWS:
    def __init__(self,portno,oe):
        self.mainthread = None
        self.server = None
        self.portno = portno
        self.oe = oe

    def ThreadFunc(self):
        print("Starting Output Webservices On Port " + str(self.portno))
        info = InfoServer()
        info.model = self.oe
        self.server = server.Site(info)
        reactor.listenTCP(int(self.portno), self.server)
        reactor.run(installSignalHandlers=False)

    def Start(self):
        self.mainthread = threading.Thread(target=self.ThreadFunc)
        self.mainthread.start()

    def Stop(self):
        if self.mainthread is not None:
            reactor.stop()
            pass
        self.mainthread = None

    